<?php

namespace App\Controller;

use App\Controller\Base\AbstractInvoiceController;
use App\Entity\Model\Customer;
use App\Entity\Model\Empresa;
use App\Entity\Model\Invoice;
use App\Entity\Model\Item;
use App\Entity\Model\ItemPago;
use App\Entity\Model\PlacasSocios;
use App\Entity\Model\Tax;
use App\Form\CustomerType;
use App\Form\InvoiceType;
use App\Form\ListGenericType;
use App\Form\SearchGenericType;
use App\Service\EmFactory;
use App\Util\ExportInvoicePartnerExcel;
use App\Util\Funciones;
use App\Util\RestApiFunciones;
use Doctrine\ORM\EntityManagerInterface;
use Knp\Component\Pager\PaginatorInterface;
use Mobile_Detect;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Psr\Log\LoggerInterface;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\Extension\Core\Type\HiddenType;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Contracts\Translation\TranslatorInterface;

/**
 * @Route("/invoice")
 */
class InvoiceController extends AbstractInvoiceController
{

    public $translator;

    /**
     * @Route("", name="invoice_index")
     *
     */
    public function indexAction(EmFactory $emFactory, Request $request, PaginatorInterface $paginator, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(Empresa::class);
        $emisor = $empresaRepo->findOneByUser($user->getRuc());

        $repo = $em->getRepository(Invoice::class);
        $repo->setPaginator($paginator);
        // @todo Unhardcode this.
        $limit = 20;

        if ($request->query->has('pdf') || $request->query->has('excel'))
            $limit = 10000;

        /*$fecha = new \DateTime();
        $fecha->modify('first day of this month');
        $desde = \DateTime::createFromFormat('d/m/Y', $fecha->format('d/m/Y'));
        $fecha->modify('last day of this month');
        $hasta = \DateTime::createFromFormat('d/m/Y', $fecha->format('d/m/Y'));

        $data = [
            'terms' => null,
            'status' => null,
            'date_from' => $desde,
            'date_to' => $hasta
        ];

        */

        $form = $this->createForm(SearchGenericType::class, null, [
            'action' => $this->generateUrl('invoice_index'),
            'method' => 'GET',
        ]);
        $form->handleRequest($request);

        $desdeFecha = null;
        $hastaFecha = null;

        if ($form->isSubmitted() && $form->isValid()) {
            $pagination = $repo->paginatedSearch($form->getData(), $limit, $request->query->getInt('page', 1), $emisor->getId());

            foreach ($form->getData() as $field => $value) {
                if ($value === null) {
                    continue;
                }
                if ($field == 'date_from') {
                    $desdeFecha = $value;
                } elseif ($field == 'date_to') {
                    $hastaFecha = $value;
                }
            }

        } else {
            $pagination = $repo->paginatedSearch([], $limit, $request->query->getInt('page', 1), $emisor->getId());
        }

        $invoices = [];
        $reporte = [];
        foreach ($pagination->getItems() as $item) {
            $invoices[] = $item[0];

            if ($request->query->has('pdf') || $request->query->has('excel')){
                //$item[0] = new Invoice();
                $aux['emision'] = $item[0]->getIssueDate();
                $aux['serie'] = $item[0]->getSerie();
                $aux['numero'] = $item[0]->getNumber();
                $aux['ruc'] = $item[0]->getCustomerIdentification();
                $aux['cliente'] = $item[0]->getCustomerName();
                $aux['basedoce'] = $item[0]->getBaseiva();
                $aux['basecero'] = $item[0]->getBasecero();
                $aux['iva'] = $item[0]->getTaxAmount();
                $aux['total'] = $item[0]->getGrossAmount();
                $aux['estado'] = $item[0]->getEstado();
                $aux['clave'] = $item[0]->getClaveAcceso();
                $aux['fecAutorizacion'] = $item[0]->getFechaAutorizacion();
                $aux['anulado'] = $item[0]->getAnulado() === true ? 'SI' : 'NO';
                $reporte[] = $aux;
            }
        }

        $listForm = $this->createForm(ListGenericType::class, $invoices, [
            'action' => $this->generateUrl('invoice_index'),
        ]);

        if ($form->isSubmitted()) {
            if ($request->query->has('pdf')) {
                return $this->reportePdf($reporte);
            } elseif ($request->query->has('excel')) {
                return $this->reporteExcel($reporte);
            }
        }

        $listForm->handleRequest($request);
        if ($listForm->isSubmitted() && $listForm->isValid()) {
            $data = $listForm->getData();
            if (empty($data['invoices'])) {
                $this->addTranslatedMessage('flash.nothing_selected', 'warning');
            } elseif ($request->request->has('delete')) {
                return $this->bulkDelete($data['invoices']);
            } elseif ($request->request->has('pdf')) {
                return $this->bulkPdf($data['invoices']);
            } elseif ($request->request->has('print')) {
                return $this->bulkPrint($data['invoices']);
            } elseif ($request->request->has('email')) {
                return $this->bulkEmail($data['invoices']);
            }
        }

        return $this->render('Invoice\index.html.twig',
            [
                'invoices' => $pagination,
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $emisor == null ? 'USD' : $emisor->getCurrency(),
                'search_form' => $form->createView(),
                'list_form' => $listForm->createView(),
            ]);
    }


    /**
     * @Route("/show/{slug}inv{id}", name="invoice_show")
     *
     */
    public function showAction($id, EmFactory $emFactory, $slug, TranslatorInterface $translator, Request $request, LoggerInterface $logger): \Symfony\Component\HttpFoundation\Response
    {
        $this->logger = $logger;
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(Empresa::class);
        $emisor = $empresaRepo->findOneByUser($user->getRuc());

        $entity = $em->getRepository(Invoice::class)->findBySlug($slug, $id);
        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Invoice entity.');
        }

        $defaultData = ['slug' => $slug, 'id'=>$id, 'customerEmail' => $entity->getCustomerEmail()];

        $form = $this->createFormBuilder($defaultData)
            ->add('slug', HiddenType::class)
            ->add('id', HiddenType::class)
            ->add('customerEmail')
            ->setAction($this->generateUrl('invoice_show', ['id' => $id, 'slug'=>$slug]))
            ->getForm();

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // data is an array with "name", "email", and "message" keys
            $data = $form->getData();

            if ($request->request->has('Form-pdf')) {
                $this->generarPdf($entity);
            } elseif ($request->request->has('Form-email')) {
                if (isset($data['customerEmail'])) {
                    $email = $data['customerEmail'];
                    $this->enviarMail($email, $entity, $em);
                } else
                    $this->addTranslatedMessage('Email del cliente nulo o en blanco', 'warning');
            } elseif ($request->request->has('Form-edit')) {
                if ($entity->getAutorizado() === false)
                    return $this->redirectToRoute('invoice_edit', ['id' => $entity->getId(), 'slug' => $slug]);
            } elseif ($request->request->has('Form-delete')) {
                $error = $this->delete($entity, $em);
                if ($error)
                    return $this->redirectToRoute('invoice_index');
            } elseif ($request->request->has('Form-anular')) {
                $entity->setAnulado(true);
                $em->persist($entity);
                $em->flush();
            } elseif ($request->request->has('Form-enviar')) {
                $this->enviarSriOnline($entity, $em);
            } elseif ($request->request->has('Form-auto')) {
                if ($entity->getAutorizado() === false) {
                    $resp = $this->consultarAutorizacion($entity, $em);
                    if ($resp != null)
                        $this->addTranslatedMessage($resp, 'danger');
                }

            }
        }

        if ($entity->getMensajeError()) {
            $this->addTranslatedMessage($entity->getMensajeError(), 'danger');
        }

        /*if (!$entity->isClosed()) {
            // When the invoice is open send to the edit form by default.
            return $this->redirect($this->generateUrl('invoice_edit', array('id' => $id)));
        }
        */

        return $this->render('Invoice\show.html.twig',
            [
                'entity' => $entity,
                'form' => $form->createView(),
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $emisor == null ? 'USD' : $emisor->getCurrency(),
                'decpunit' => $emisor->getDecPunit(),
                'transportesocio' => $entity->getTransportesocio() ? $entity->getTransportesocio() : null
            ]);
    }

    /**
     * @Route("/new", name="invoice_add")
     *
     */
    public function newAction(EmFactory $emFactory, Request $request, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        //require_once "Mobile_Detect.php";
        $detect = new Mobile_Detect;

        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $msg='';
        $puedefacturar = Funciones::getValidaPuedeFacturar($empresa, $em, $msg);
        if ($empresa->getPuedefacturar() === false && $empresa->getTipoAmbiente() === "2") {
            $this->addTranslatedMessage($msg, 'danger');
            return $this->redirect($this->generateUrl('invoice_index'));
        }

        $empresa = Funciones::getValidaEmitidos($empresa, $em);
        if ($empresa->getPuedefacturar() === false && $empresa->getTipoAmbiente() === "2") {
            $this->addTranslatedMessage('Número de comprobantes emitidos excede del plan contratado ', 'danger');
            return $this->redirect($this->generateUrl('invoice_index'));
        }


        // Check for mobile environment.
        $movil = false;
        if ($detect->isMobile()) {
            // Your code here.
            $movil = true;
        }

        $invoice = new Invoice();
        $newItem = new Item();

        $tax = $em->getRepository(Tax::class)->findTaxDefault($empresa->getId());
        $newItem->setTaxes($tax);
        $invoice->addItem($newItem);

        $itemPago = new ItemPago();
        $itemPago->setPlazo(0);

        $invoice->addPago($itemPago);

        $placas = [];

        if($empresa->getEstransportista()){
            $socio = $empresa->getTransportesocio();
            if($socio) {
                //$socio = new TransporteSocio();
                $placas = $em->getRepository('App\Entity\Model\TransporteSocio')->findPlacasSocioEmpresa($socio->getId());
            }
        }

        $form = $this->createForm(InvoiceType::class, $invoice, [
            'action' => $this->generateUrl('invoice_add'),
        ]);

        if($empresa->getEstransportista()) {
            $socio = $empresa->getTransportesocio();
            if ($socio) {
                $form->add('placa', EntityType::class, array(
                    'class' => 'App\Entity\Model\PlacasSocios',
                    'choice_label' => 'value',
                    'required' => true,
                    //'placeholder' => 'Seleccione una serie ...',
                    'label' => 'PLACA',
                    'choices' => $placas
                ));
            }
        }

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            if ($request->request->has('save_draft')) {
                $invoice->setStatus(Invoice::DRAFT);
            } else {
                // Any save action transforms this to opened.
                $invoice->setStatus(Invoice::OPENED);
            }

            //if ($invoice->getCustomer() == null && $invoice->getCustomerIdentification() != '9999999999999') {
            //    $this->addTranslatedMessage('Cliente no se encuentra registrado', 'danger');
            //} else {
            //$invoice->setNumber(null);
            $serie = $invoice->getSeries()->getValue();
            $invoice->setSerie($serie);
            $this->cargarImpuestos($invoice);
            $invoice->setEmpresa($empresa);
            $invoice->setUsuario($user->getId());
            $invoice->setAmbiente($empresa->getTipoAmbiente());

            if ($invoice->getCustomerIdentification() === '9999999999999' && floatval($invoice->getTotalFactura()) > 200) {
                $this->addTranslatedMessage('Consumidor final no puede facturar mas de 200 dolares', 'danger');

            }else {

                if ($empresa->getEstransportista()) {
                    $socio = $empresa->getTransportesocio();
                    if ($socio) {
                        $invoice->setTransportesocio($socio);
                        $auxplaca = $invoice->getPlaca()->getName();
                        $invoice->setPlaca($auxplaca);
                    }
                }

                $em->persist($invoice);
                $em->flush();
                //$this->addTranslatedMessage('flash.added');

                $invoice_id = $invoice->getId();

                $invoice = $this->generarXml($invoice, $em);

                if ($empresa->getEnvioAutomatico()) {

                    /*if ($invoice !== null) {
                        $app_url = $this->getParameter('cola_url');

                        $error = false;

                        $resp = RestApiFunciones::enviarComprobanteCola($error,
                            $invoice->getCustomerName(),
                            $invoice->getCustomerEmail(),
                            $user->getId(),
                            $invoice->getXml(),
                            $invoice->getClaveAcceso(),
                            $app_url,
                            'invoice'
                        );

                        if ($error) {
                            $this->addTranslatedMessage($resp, 'danger');
                            $invoice->setMensajeError($resp);
                        }

                        $em->persist($invoice);
                        $em->flush();

                    }

                    return $this->redirect($this->generateUrl('invoice_index'));
                    */

                }

                return $this->redirectToRoute('invoice_show', ['id' => $invoice_id, 'slug' => $invoice->getSlug()]);
                //}
            }
        }

        $customer = new Customer();

        $formcustomer = $this->createForm(CustomerType::class, $customer, [
            'action' => $this->generateUrl('rest_customer_add'),
        ]);
        $formcustomer->handleRequest($request);

        return $this->render('Invoice\edit.html.twig',
            [
                'form' => $form->createView(),
                'movil' => $movil,
                'formcustomer' => $formcustomer->createView(),
                'customer' => $customer,
                'entity' => $invoice,
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $empresa == null ? 'USD' : $empresa->getCurrency(),
                'decpunit' => $empresa->getDecPunit(),
                'estransportista' => $empresa->getEstransportista()
            ]);
    }

    /**
     * @Route("/edit/{slug}inv{id}", name="invoice_edit")
     *
     */
    public function editAction($id, EmFactory $emFactory, Request $request, $slug, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $entity = $em->getRepository(Invoice::class)->findBySlug($slug, $id);
        if (!$entity && $entity->getEmpresa() != $empresa) {
            throw $this->createNotFoundException('Unable to find Invoice entity.');
        }

        if($entity->getAutorizado()){
            return $this->redirectToRoute('invoice_show', ['id' => $id, 'slug' => $slug]);
        }

        $placas = [];

        if($empresa->getEstransportista()){
            $socio = $empresa->getTransportesocio();
            if($socio) {
                //$socio = new TransporteSocio();
                $placas = $placas = $em->getRepository('App\Entity\Model\TransporteSocio')->findPlacasSocioEmpresa($socio->getId());
            }
        }

        $form = $this->createForm(InvoiceType::class, $entity, [
            'action' => $this->generateUrl('invoice_edit', ['id' => $id, 'slug' => $slug]),
        ]);

        if($empresa->getEstransportista()) {
            $socio = $empresa->getTransportesocio();
            if ($socio) {
                $form->add('placa', EntityType::class, array(
                    'class' => 'App\Entity\Model\PlacasSocios',
                    'choice_label' => 'value',
                    'required' => true,
                    //'placeholder' => 'Seleccione una serie ...',
                    'label' => 'PLACA',
                    'choices' => $placas,
                    'choice_value' => function ($value) {
                        //return $value->getName();

                        if(gettype($value) === PlacasSocios::class)
                            return $value->getName();
                        else
                            return  $value;
                    },
                ));
            }
        }

        $form->handleRequest($request);

        $entity->setEmpresa($empresa);

        if ($form->isSubmitted() && $form->isValid()) {
            $redirectRoute = 'invoice_show';

            $serie = $entity->getSeries()->getValue();
            $entity->setSerie($serie);
            $entity->setUsuario($user->getId());
            $this->cargarImpuestos($entity);

            if ($request->request->has('save_draft')) {
                $entity->setStatus(Invoice::DRAFT);
            } elseif ($request->request->has('save_close')) {
                $entity->setForcefullyClosed(true);
            } elseif ($entity->isDraft()) {
                // Any save action transforms this to opened.
                $entity->setStatus(Invoice::OPENED);
            }

            if ($empresa->getEstransportista()) {
                $socio = $empresa->getTransportesocio();
                if ($socio) {
                    $entity->setTransportesocio($socio);
                    $auxplaca = $entity->getPlaca()->getName();
                    $entity->setPlaca($auxplaca);
                }
            }

            if ($entity->getCustomerIdentification() === '9999999999999' && floatval($entity->getTotalFactura()) > 200) {
                $this->addTranslatedMessage('Consumidor final no puede facturar mas de 200 dolares', 'danger');

            } else {

                // Save.
                $em->persist($entity);
                $em->flush();
                //$this->addTranslatedMessage('flash.updated');

                $entity = $this->generarXml($entity, $em);

                return $this->redirectToRoute($redirectRoute, ['id' => $id, 'slug' => $slug]);

            }
        }

        $customer = new Customer();

        $formcustomer = $this->createForm(CustomerType::class, $customer, [
            'action' => $this->generateUrl('rest_customer_add'),
        ]);
        $formcustomer->handleRequest($request);

        return $this->render('Invoice\edit.html.twig',
            [
                'entity' => $entity,
                'form' => $form->createView(),
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $empresa == null ? 'USD' : $empresa->getCurrency(),
                'decpunit' => $empresa->getDecPunit(),
                'formcustomer' => $formcustomer->createView(),
                'estransportista' => $empresa->getEstransportista()
            ]);
    }

    public function enviarSriOnline(Invoice $invoice, EntityManagerInterface $em)
    {
        $invoice = $this->generarXml($invoice, $em);

        if($invoice !== null) {
            $app_url = $this->getParameter('api_url');

            $error = false;
            $resp = RestApiFunciones::enviarComprobante($error, $invoice->getXml(), $invoice->getClaveAcceso(), $app_url);

            if ($error) {
                $this->addTranslatedMessage($resp->message, 'danger');
            } elseif ($resp->estado === 'recibido') {
                $resp = $this->consultarAutorizacion($invoice, $em);
                if ($resp !== null)
                    $this->addTranslatedMessage($resp, 'danger');
            } else {
                $invoice->setMensajeError($resp->message);
                $this->addTranslatedMessage($resp->message, 'danger');
                $em->persist($invoice);
                $em->flush();
            }

        }
    }

    /**
     * @Route("/payments/{slug}inv{id}", name="invoice_payments")
     *
     */
    public function paymentsAction(Request $request, $slug, EmFactory $emFactory, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $emisor = null;
        $this->translator = $translator;

        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        // Return all payments
        $invoice = $em->getRepository(\App\Entity\Model\Invoice::class)->findBySlug($slug);
        if (!$invoice) {
            throw $this->createNotFoundException('Unable to find Invoice entity.');
        }

        $payment = new Payment;
        $addForm = $this->createForm(\App\Form\PaymentType::class, $payment, [
            'action' => $this->generateUrl('invoice_payments', ['id' => $invoice->getId(), 'slug' => $slug]),
        ]);
        $addForm->handleRequest($request);
        if ($addForm->isSubmitted() && $addForm->isValid()) {
            $invoice->addPayment($payment);
            $em->persist($invoice);
            $em->flush();
            $this->addTranslatedMessage('payment.flash.added');

            // Rebuild the query, since we have new objects now.
            return $this->redirectToRoute('invoice_index');
        }

        $listForm = $this->createForm('App\Form\ListInvoicePaymentType', $invoice->getPayments()->getValues(), [
            'action' => $this->generateUrl('invoice_payments', ['id' => $invoice->getId(), 'slug' => $slug]),
        ]);
        $listForm->handleRequest($request);

        if ($listForm->isSubmitted() && $listForm->isValid()) {
            $data = $listForm->getData();
            foreach ($data['payments'] as $payment) {
                $invoice->removePayment($payment);
                $em->persist($invoice);
                $em->flush();
            }

            $this->addTranslatedMessage('payment.flash.bulk_deleted');

            // Rebuild the query, since some objects are now missing.
            return $this->redirectToRoute('invoice_index');
        }

        return $this->render('Payment\list.html.twig',
            [
                'invoiceId' => $invoiceId,
                'add_form' => $addForm->createView(),
                'list_form' => $listForm->createView(),
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $emisor == null ? 'USD' : $emisor->getCurrency(),
            ]);
    }

    /**
     * @Route("/form-totals", name="invoice_form_totals")
     */
    public function getInvoiceFormTotals(EmFactory $emFactory, Request $request): \Symfony\Component\HttpFoundation\Response
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();

        $post = $request->request->get('invoice');
        if (!$post) {
            throw new NotFoundHttpException;
        }

        $response = $this->getInvoiceTotalsFromPost($post, new Invoice, $request->getLocale(), $em);

        return new JsonResponse($response);
    }

    /**
     * @Route("/pdfpreview/{slug}inv{id}", name="invoice_show_pdf_preview")
     *
     */
     
    
     
     
     
     
    public function showOnlinePdfAction($id,  $slug, EmFactory $emFactory, TranslatorInterface $translator, Request $request, LoggerInterface $logger): \Symfony\Component\HttpFoundation\Response
    {
        $this->logger = $logger;
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $invoice = $em->getRepository(\App\Entity\Model\Invoice::class)->findBySlug($slug, $id);
        if (!$invoice) {
            throw $this->createNotFoundException('Unable to find Invoice entity.');
        }

        $filename = 'FAC_' . $invoice->getSerie() . "-" . str_pad($invoice->getNumber(), 9, '0', STR_PAD_LEFT) . '.pdf';

        $xml = $invoice->getAutorizado() ? $invoice->getXmlAutorizado() : $invoice->getXml();

        $app_url = $this->getParameter('api_url').'facturaride.php';
        
        

        $error = false;
        $mensaje = "";
        $logo = null;
        if($empresa->getRutaLogo()) {
            if (strlen(trim($empresa->getRutaLogo())) > 0) {
                $logoaux = trim($empresa->getRutaLogo());
                $logoaux = explode("/", $logoaux);
                if(sizeof($logoaux) === 2){
                    $logo = $logoaux[1];
                }
            }
        }
        $docPdf = RestApiFunciones::getPdf($error, $app_url, $invoice->getClaveAcceso(), $xml, $mensaje, $logo);

        if ($error) {
            $this->addTranslatedMessage('ERRROR AL GENERAR EL PDF, ' . $mensaje, 'danger');
        } else {
            file_put_contents($filename, $docPdf);

            $pdf = base64_encode(file_get_contents($filename));

            $response = new Response($pdf);
            $response->headers->set('Content-Type', 'application/octet-stream');
            $response->headers->set('Content-Description', 'File Transfer');
            $response->headers->set('Content-Disposition', 'attachment; filename="'.$filename.'"');
            // $response->headers->set('Expires', '0');
            // $response->headers->set('Content-Transfer-Encoding', 'binary');
            $response->headers->set('Content-length', strlen($pdf));
            $response->headers->set('Cache-Control', 'no-cache private');
            // $response->headers->set('Pragma', 'public');
            // Send headers before outputting anything
            $response->sendHeaders();



            try {
                unlink($filename);
            } catch (\Exception $exception) {

            }

            return $response;
        }
    }




    
    
    public static function showOnlinePdfActionAPI($id,  $slug, EmFactory $emFactory, Request $request, $user, $xmlapi, $app_url ): string
    {



        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $invoice = $em->getRepository(\App\Entity\Model\Invoice::class)->findBySlug($slug, $id);


        $filename = 'FAC_' . $invoice->getSerie() . "-" . str_pad($invoice->getNumber(), 9, '0', STR_PAD_LEFT) . '.pdf';
        
        $xml = $invoice->getAutorizado() ? $xmlapi : $invoice->getXml();

        

        //$app_url = 'https://api.countelseg.online/'.'facturaride.php';


        $error = false;
        $mensaje = "";
        $logo = null;
        if($empresa->getRutaLogo()) {
            if (strlen(trim($empresa->getRutaLogo())) > 0) {
                $logoaux = trim($empresa->getRutaLogo());
                $logoaux = explode("/", $logoaux);
                if(sizeof($logoaux) === 2){
                    $logo = $logoaux[1];
                }
            }
        }
        $docPdf = RestApiFunciones::getPdf($error, $app_url, $invoice->getClaveAcceso(), $xml, $mensaje, $logo);

        if ($error) {
            $mensaje = "Error al obtener PDF";
        } else {
            file_put_contents($filename, $docPdf);

            $pdf = base64_encode(file_get_contents($filename));

            $response = base64_encode(file_get_contents($filename));


            

            return $response;
            
            try {
                unlink($filename);
            } catch (\Exception $exception) {

            }



        }
    }
     




















    /**
     * @Route("download/xml/{slug}inv{id}", name="invoice_download_xml")
     *
     */
    public function xmldownloadAction($id, $slug, EmFactory $emFactory, Request $request, TranslatorInterface $translator, LoggerInterface $logger)
    {
        $this->logger = $logger;
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $invoice = $em->getRepository('App\Entity\Model\Invoice')->findBySlug($slug, $id);
        if (!$invoice) {
            throw $this->createNotFoundException('Unable to find Invoice entity.');
        }
        $filename = 'FAC_' . $invoice->getSerie() . "-" . str_pad($invoice->getNumber(), 9, '0', STR_PAD_LEFT) . '.xml';

        try {
            $xml = $invoice->getAutorizado() ? $invoice->getXmlAutorizado() : $invoice->getXml();

            file_put_contents($filename, $xml);

            $contentType = 'application/xml';

            $content = file_get_contents($filename);
            $response = new Response();
            $response->headers->set('Content-Type', $contentType);
            $response->headers->set('Content-Disposition', 'attachment;filename="' . $filename . '"');

            $response->headers->addCacheControlDirective('no-cache', true);
            $response->headers->addCacheControlDirective('must-revalidate', true);

            $response->setContent($content);
            return $response;
        } catch (\Exception $ex) {

        } finally {
            try {
                unlink($filename);
            } catch (\Exception $ex) {

            }
        }
    }

    private function reporteExcel($invoices){

        $contentType = null;
        $format = 'xlsx';
        $filename = 'facturas' . '.' . $format;

        $exportExcel = new ExportInvoicePartnerExcel();

        $titulo = "REPORTE FACTURAS";
        $desde = null;
        $hasta = null;

        $spreadsheet = $exportExcel->createSpreadsheet($invoices, $titulo, $desde, $hasta);

        if ($format === 'xlsx') {
            $contentType = 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet';
            $writer = new Xlsx($spreadsheet);
        }

        //$writer->save($filename);

        $response = new StreamedResponse();
        $response->headers->set('Content-Type', $contentType);
        $response->headers->set('Content-Disposition', 'attachment;filename="'.$filename.'"');

        $response->setPrivate();
        $response->headers->addCacheControlDirective('no-cache', true);
        $response->headers->addCacheControlDirective('must-revalidate', true);

        $response->setCallback(static function () use ($writer) {
            $writer->save('php://output');
        });

        return $response;

    }


}
