<?php

namespace App\Controller;

use App\Entity\Model\Factura;
use App\Form\SearchFacturaType;
use App\Form\FacturaType;
use App\Repository\EmpresaRepository;
use App\Util\ExportInvoicePartnerExcel;
use App\Util\RestApiFunciones;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\QueryBuilder;
use Knp\Component\Pager\PaginatorInterface;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @Route("/factura/demo")
 */
class FacturaController extends AbstractController
{
    private  $estado = "";

    /**
     * @Route("/", name="factura_index", methods={"GET","POST"})
     */
    public function index(EmpresaRepository $empresaRepository, Request $request, PaginatorInterface $paginator, EntityManagerInterface $entityManager): Response
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();
        $empresa = RestApiFunciones::getEmpresa($empresaRepository, $user->getRucEmpresa());

        $formSearch = $this->createForm(SearchFacturaType::class, null, [
            'action' => $this->generateUrl('factura_index'),
            'method' => 'GET',
        ]);
        $formSearch->handleRequest($request);

        if (!$empresa) {
            $this->addFlash('danger', 'Datos de la empresa no existe, RUC:' . $user->getRucEmpresa());
            $query = [];
        } else {
            $em = $entityManager->getManager($empresa->getDataBase());

            $query = $em->createQueryBuilder('p');
            $query->from(Factura::class, 'p');
            $query->select("p.id, p.emision, p.serie, p.numero, p.ruc, p.cliente, p.estado, p.clave, p.fecAutorizacion");

            if ($formSearch->isSubmitted() && $formSearch->isValid()) {
                $query = $this->applySearchParamsToQuery($formSearch->getData(), $query);
            }

            $query->orderBy('p.numero', 'asc');
            $query->getQuery();
        }

        $facturas = RestApiFunciones::getPaginationResults($query, $paginator, $request, $this->getParameter('limit_pagination'));

        return $this->render('factura/index.html.twig', [
            'facturas' => $facturas,
            'estado' => $this->estado,
            'searchform' => $formSearch->createView(),
        ]);
    }

    /**
     * @Route("/downloadxls", name="factura_downloadxls", methods={"POST"})
     */
    public function downloadXls(EmpresaRepository $empresaRepository, Request $request, EntityManagerInterface $entityManager) :Response
    {
        $contentType = null;
        $error = "";
        try {
            $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
            $user = $this->getUser();
            $empresa = RestApiFunciones::getEmpresa($empresaRepository, $user->getRucEmpresa());

            $formSearch = $this->createForm(SearchFacturaType::class, null, [
                'action' => $this->generateUrl('factura_index'),
                'method' => 'POST',
            ]);
            $formSearch->handleRequest($request);

            if ($empresa) {
                $em = $entityManager->getManager($empresa->getDataBase());

                $query = $em->createQueryBuilder('p');
                $query->from(Factura::class, 'p');
                $query->select("p.emision, p.serie, p.numero, p.ruc, p.cliente, p.estado, p.clave, p.fecAutorizacion");

                if ($formSearch->isSubmitted() && $formSearch->isValid()) {
                    $query = $this->applySearchParamsToQuery($formSearch->getData(), $query);
                }

                $query->orderBy('p.numero', 'asc');

                $invoices = $query->getQuery()->getArrayResult();

                $format = 'xlsx';
                $filename = 'facturas' . '.' . $format;

                $exportExcel = new ExportInvoicePartnerExcel();

                $titulo = "REPORTE FACTURAS";

                $spreadsheet = $exportExcel->createSpreadsheet($invoices, $titulo);

                if ($format === 'xlsx') {
                    $contentType = 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet';
                    $writer = new Xlsx($spreadsheet);
                }

                //$writer->save($filename);

                $response = new StreamedResponse();
                $response->headers->set('Content-Type', $contentType);
                $response->headers->set('Content-Disposition', 'attachment;filename="'.$filename.'"');
                $response->setPrivate();
                $response->headers->addCacheControlDirective('no-cache', true);
                $response->headers->addCacheControlDirective('must-revalidate', true);
                $response->setCallback(static function () use ($writer) {
                    $writer->save('php://output');
                });

                return $response;
            }
        } catch (\Exception $exception) {
            $error = $exception->getMessage();
        }

        $response = new JsonResponse($error, \Symfony\Component\HttpFoundation\Response::HTTP_ACCEPTED);
        $response->setData(['error' => true, 'mensaje' => $error]);

        return $response;

    }

    protected function applySearchParamsToQuery(array $params, QueryBuilder $qb)
    {
        foreach ($params as $field => $value) {
            if ($value === null) {
                continue;
            }

            if ($field == 'terms') {
                $terms = $qb->expr()->literal(sprintf('%%%s%%', $value));
                    $expr = $qb->expr()->orX(
                        $qb->expr()->like('p.cliente', $terms)
                    );

                $qb->andWhere($expr);
                $qb->orWhere('p.numero = :numero');
                $qb->setParameter('numero', $value);

            } elseif ($field == 'date_from') {
                $qb->andWhere('p.emision >= :date_from');
                $qb->setParameter('date_from', $value);
            } elseif ($field == 'date_to') {
                $qb->andWhere('p.emision <= :date_to');
                $qb->setParameter('date_to', $value);
            } elseif ($field == 'status') {
                $qb->andWhere('p.estado = :status');
                $qb->setParameter('status', $value);
                $this->estado = $value;
            }
        }

        return $qb;
    }


    /**
     * @Route("/new", name="factura_new", methods={"GET","POST"})
     */
    public function new(Request $request, EntityManagerInterface $entityManager): Response
    {
        $factura = new Factura();
        $form = $this->createForm(FacturaType::class, $factura);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            $entityManager->persist($factura);
            $entityManager->flush();

            return $this->redirectToRoute('factura_index');
        }

        return $this->render('factura/new.html.twig', [
            'factura' => $factura,
            'form' => $form->createView(),
        ]);
    }

    /**
     * @Route("/{id}", name="factura_show", methods={"GET"})
     */
    public function show(Factura $factura): Response
    {
        return $this->render('factura/show.html.twig', [
            'factura' => $factura,
        ]);
    }

    /**
     * @Route("/{id}/edit", name="factura_edit", methods={"GET","POST"})
     */
    public function edit(Request $request, Factura $factura, EntityManagerInterface $entityManager): Response
    {
        $form = $this->createForm(FacturaType::class, $factura);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->flush();

            return $this->redirectToRoute('factura_index');
        }

        return $this->render('factura/edit.html.twig', [
            'factura' => $factura,
            'form' => $form->createView(),
        ]);
    }

    /**
     * @Route("/{id}", name="factura_delete", methods={"DELETE"})
     */
    public function delete(Request $request, Factura $factura, EntityManagerInterface $entityManager): Response
    {
        if ($this->isCsrfTokenValid('delete'.$factura->getId(), $request->request->get('_token'))) {

            $entityManager->remove($factura);
            $entityManager->flush();
        }

        return $this->redirectToRoute('factura_index');
    }
}
