<?php

namespace App\Repository;

use App\Entity\Model\Empresa;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\QueryBuilder;
use Doctrine\Common\Collections\ArrayCollection;
use Knp\Component\Pager\PaginatorInterface;
use App\Entity\Model\Invoice;
use App\Entity\Model\Item;
use App\Entity\Model\Series;

/**
 * Repository class to be inherited by InvoiceRepository,
 * RecurringInvoiceRepository and EstimateRepository
 */
class AbstractInvoiceRepository extends EntityRepository
{

    public $empresa_id;

    /**
     * @var mixed|\Knp\Component\Pager\PaginatorInterface
     */
    public $paginator;

    public function findBySlug(string $slug, $id) : ?Invoice
    {
        try {
            return $this->getEntityManager()
                ->createQueryBuilder()
                ->select('c')
                ->from(Invoice::class, 'c')
                ->Where("c.slug = :slug")
                ->andWhere("c.id = :id")
                ->setParameter('slug', $slug)
                ->setParameter('id', $id)
                ->getQuery()
                ->getOneOrNullResult();
        } catch (NonUniqueResultException $nonUniqueResultException) {
            return null;
        }
    }

    /**
     * Finds an entity by its primary key / identifier.
     *
     * @param $id.
     * @param $empresa_id
     * @return object|null The entity instance or NULL if the entity can not be found.
     */
    public function findId($id, $empresa_id)
    {
        $class = $this->getEntityName();

        return $this->getEntityManager()->createQueryBuilder()
            ->select('i')
            ->from($class, 'i')
            ->where('i.id = :id')
            ->andWhere('i.empresa = :empresa_id')
            ->setParameter('id', $id)
            ->setParameter('empresa_id', $empresa_id)
            ->getQuery()
            ->getOneOrNullResult();
    }

    /**
     * getNextNumber
     * Obtain the next numer available for the provided series
     * @param \App\Entity\Model\Series $serie
     * @return integer
     */
    public function getNextNumber(Series $series, $empresa_id)
    {
        $class = $this->getEntityName();
        $result = $this->getEntityManager()->createQueryBuilder()
            ->select('MAX(i.number) AS max_number')
            ->from($class, 'i')
            //->where('i.status <> :status')
            ->where('i.series = :series')
            ->andWhere('i.empresa = :empresa')
            //->setParameter('status', $class::DRAFT)
            ->setParameter('series', $series)
            ->setParameter('empresa', $empresa_id)
            ->getQuery()
            ->getSingleResult();

        return empty($result['max_number'])
            ? $series->getIniFactura()
            : $result['max_number'] + 1;
    }

    /**
     * getNextNumber
     * Obtain the next numer available for the provided series
     * @param String $serie
     * @return integer
     */
    public function getNextNumberAmbiente($ambiente)
    {
        $class = $this->getEntityName();
        $result = $this->getEntityManager()->createQueryBuilder()
            ->select('MAX(i.number) AS max_number')
            ->from($class, 'i')
            ->where('i.ambiente = :ambiente')
            ->setParameter('ambiente', $ambiente)
            ->getQuery()
            ->getSingleResult();

        return empty($result['max_number'])
            ? 1
            : $result['max_number'] + 1;
    }

    /**
     * Update totals for invoices, recurring or estimates
     * @param ArrayCollection of entities
     * @return AbstractInvoiceRepository
     **/
    public function updateTotals()
    {
        $em = $this->getEntityManager();
        foreach ($em->createQuery('SELECT i from '.$this->getEntityName().'  i')->getResult() as $entity) {
            $entity->checkAmounts();
            $em->persist($entity);
        }

        $em->flush();
        return $this;
    }


    public function findByItem(Item $item)
    {
        $qb = $this->getEntityManager()->createQueryBuilder();
        $qb->select('i')
            ->from($this->getEntityName(), 'i')
            ->join('i.items', 'ii')
            ->where('ii.id = ?1')
            ->setParameter(1, $item->getId());

        return $qb->getQuery()->getResult();
    }

    public function paginatedSearch(array $params, $limit, $page, $empresa_id)
    {
        $this->empresa_id = $empresa_id;

        if (!$this->paginator) {
            throw new \RuntimeException('You have to set a paginator.yaml first using setPaginator() method');
        }

        $qb = $this->getEntityManager()
            ->createQueryBuilder()
            ->from($this->getEntityName(), 'i')
            ->where('i.empresa = :empresa')
            ->setParameter('empresa', $empresa_id);

        $this->addPaginatedSearchSelects($qb);
        $this->applySearchParamsToQuery($params, $qb);

        return $this->paginator->paginate($qb->getQuery(), $page, $limit, [
            'defaultSortFieldName' => 'i.id',
            'defaultSortDirection' => 'desc',
        ]);
    }

    /**
     * There is no easy way to inject things into repositories yet.
     */
    public function setPaginator(PaginatorInterface $paginator)
    {
        $this->paginator = $paginator;
    }

    public function getTotals(array $params, $empresa_id)
    {
        $qb = $this->getEntityManager()
            ->createQueryBuilder()
            ->from($this->getEntityName(), 'i');

        $qb->where('i.empresa = :empresa');
        $qb->setParameter('empresa', $empresa_id);

        $this->applySearchParamsToQuery($params, $qb);

        $qb->addSelect('SUM(i.gross_amount)');
        $qb->addSelect('SUM(i.paid_amount)');
        $qb->addSelect('SUM(i.gross_amount - i.paid_amount)');
        $qb->addSelect('SUM(i.net_amount)');
        $qb->addSelect('SUM(i.tax_amount)');
        if (!empty($params['tax'])) {
            // Tax's total cost.
            $qb->addSelect('SUM(it.unitary_cost * (tx.value/100))');
        }

        $qb->setMaxResults(1);

        $result = $qb->getQuery()->getSingleResult();

        // Transform to named array for easier access.
        $totals = [
            'gross' => $result[1],
            'paid' => $result[2],
            'due' => $result[3],
            'net' => $result[4],
            'tax' => $result[5],
        ];

        if (!empty($params['tax'])) {
            $totals['tax_' . $params['tax']] = $result[6] ?? 0;
        }

        return $totals;
    }

    protected function applySearchParamsToQuery(array $params, QueryBuilder $qb)
    {
        foreach ($params as $field => $value) {
            if ($value === null) {
                continue;
            }

            if ($field == 'terms') {
                 $terms = $qb->expr()->literal(sprintf('%%%s%%', $value));
                if(is_numeric($value))
                    $expr = $qb->expr()->like('i.number', $terms);
                else
                    $expr = $qb->expr()->like('i.customer_name', $terms);
                $qb->andWhere($expr);
            } elseif ($field == 'date_from') {
                $qb->andWhere('i.issue_date >= :date_from');
                $qb->setParameter('date_from', $value);
            } elseif ($field == 'date_to') {
                $qb->andWhere('i.issue_date <= :date_to');
                $qb->setParameter('date_to', $value);
            } elseif ($field == 'status') {
                $autorizado = false;
                if($value == 'autorizado') {
                    $autorizado = true;
                }

                $qb->andWhere('i.autorizado = :autorizado');
                $qb->setParameter('autorizado', $autorizado);
            } elseif ($field == 'customer') {
                $customer = $qb->expr()->literal(sprintf('%%%s%%', $value));
                $qb->andWhere($qb->expr()->orX(
                    $qb->expr()->like('i.customer_name', $customer),
                    $qb->expr()->like('i.customer_identification', $customer)
                ));
            } elseif ($field == 'series') {
                $qb->andWhere('i.series = :series');
                $qb->setParameter('series', $value);
            } elseif ($field == 'tax') {
                $qb->join('i.items', 'it');
                $qb->join('it.taxes', 'tx');
                $qb->andWhere('tx.id = :tax');
                $qb->setParameter('tax', $value);
            }
        }
    }

    protected function addPaginatedSearchSelects(QueryBuilder $qb)
    {
        // Select everything by default.
        $qb->select('i');
    }

    public function countEmitidos($empresa_id){

        return $this->getEntityManager()
            ->createQueryBuilder()
            ->select('count(c.id) ')
            ->from(Invoice::class, 'c')
            ->where('c.empresa = :empresa_id')
            ->setParameter('empresa_id', $empresa_id)
            ->getQuery()
            ->getSingleScalarResult();

    }

    public function paginatedSearchReembolso(array $params, $limit, $page, $empresa_id)
    {
        $this->empresa_id = $empresa_id;

        if (!$this->paginator) {
            throw new \RuntimeException('You have to set a paginator.yaml first using setPaginator() method');
        }

        $qb = $this->getEntityManager()
            ->createQueryBuilder()
            ->from($this->getEntityName(), 'i')
            ->where('i.empresa = :empresa')
            ->andWhere('i.esreembolso = true')
            ->setParameter('empresa', $empresa_id);

        $this->addPaginatedSearchSelects($qb);
        $this->applySearchParamsToQuery($params, $qb);

        return $this->paginator->paginate($qb->getQuery(), $page, $limit, [
            'defaultSortFieldName' => 'i.id',
            'defaultSortDirection' => 'desc',
        ]);
    }

}
