<?php

namespace App\Entity\Model;

use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\PasswordHasher\Hasher\PasswordHasherAwareInterface;
use Symfony\Component\Security\Core\User\PasswordAuthenticatedUserInterface;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Validator\Constraints as Assert;


/**
 * @ORM\Entity(repositoryClass="App\Repository\UserRepository")
 * @UniqueEntity(fields={"username"}, message="Usuario ya se encuentra registrado")
 * @UniqueEntity(fields={"email"}, message="Email ya se encuentra registrado")
 */
class User implements UserInterface, PasswordAuthenticatedUserInterface, PasswordHasherAwareInterface
{
    /**
     * @ORM\Id()
     * @ORM\GeneratedValue()
     * @ORM\Column(type="integer")
     */
    private $id;

    /**
     * @ORM\Column(type="string", length=180, unique=true)
     * @Assert\NotBlank(message="Email es requerido")
     * @Assert\Email(message="Email no valido")
     */
    private $email;

    /**
     * @ORM\Column(type="string", length=13, unique=true)
     * @Assert\NotBlank(message="Usuario es requerido")
     */
    private $username;

    /**
     * @ORM\Column(type="string", length=200, nullable=true)
     * @Assert\NotBlank(message="Razón Social es requerido")
     */
    private $razonSocial;

    /**
     * @ORM\Column(type="string", length=13, nullable=true)
     * @Assert\NotBlank(message="RUC es requerido")
     * @Assert\Length(min=13, max=13, exactMessage="numero de ruc debe contener 13 digitos")
     */
    private $ruc;

    /**
     * @ORM\Column(type="json")
     */
    private  $roles = [];

    /**
     * @var string The hashed password
     * @ORM\Column(type="string")
     */
    private $password = null;

    /**
     * @ORM\Column(type="string", length=100)
     */
    private $dataBase = null;

    /**
     * @Gedmo\Timestampable(on="create")
     * @ORM\Column(type="datetime")
     */
    private $create_date;

    /**
     * @ORM\Column(type="string", length=200)
     */
    private $svslug = null;

    /**
     * @ORM\Column(name="plan_id", type="integer", nullable=true)
     */
    private  $plan_id = null;

    /**
     * User constructor.
     */
    public function __construct()
    {
        $this->create_date = new \DateTime('now');
    }


    public function getId()
    {
        return $this->id;
    }

    /**
     * A visual identifier that represents this user.
     *
     * @see UserInterface
     */
    public function getUsername(): string
    {
        return (string) $this->username;
    }

    public function setUsername(string $username): self
    {
        $this->username = $username;

        return $this;
    }

    /**
     * @see UserInterface
     */
    public function getRoles(): array
    {
        $roles = $this->roles;
        // guarantee every user at least has ROLE_USER
        $roles[] = 'ROLE_USER';

        return array_unique($roles);
    }

    public function setRoles(array $roles): self
    {
        $this->roles = $roles;

        return $this;
    }

    /**
     * @see UserInterface
     */
    public function getPassword(): string
    {
        return (string) $this->password;
    }

    public function setPassword(string $password): self
    {
        $this->password = $password;

        return $this;
    }

    /**
     * @return mixed
     */
    public function getEmail()
    {
        return $this->email;
    }

    /**
     * @param mixed $email
     */
    public function setEmail($email): void
    {
        $this->email = $email;
    }


    /**
     * @see UserInterface
     */
    public function getSalt()
    {
        return "";
    }

    /**
     * @see UserInterface
     */
    public function eraseCredentials()
    {
        // If you store any temporary, sensitive data on the user, clear it here
        // $this->plainPassword = null;
    }

    /**
     * @return mixed
     */
    public function getRazonSocial()
    {
        return $this->razonSocial;
    }

    /**
     * @param mixed $razonSocial
     */
    public function setRazonSocial($razonSocial): void
    {
        $this->razonSocial = $razonSocial;
    }

    /**
     * @return string
     */
    public function getDataBase()
    {
        return $this->dataBase;
    }

    /**
     * @param string $dataBase
     */
    public function setDataBase(string $dataBase): void
    {
        $this->dataBase = $dataBase;
    }

    /**
     * @return mixed
     */
    public function getRuc()
    {
        return $this->ruc;
    }

    /**
     * @param mixed $ruc
     */
    public function setRuc($ruc): void
    {
        $this->ruc = $ruc;
    }

    /**
     * @return mixed
     */
    public function getPlanId()
    {
        return $this->plan_id;
    }

    /**
     * @param int $plan_id
     */
    public function setPlanId(int $plan_id)
    {
        $this->plan_id = $plan_id;
    }


    public function __call($name, $arguments)
    {
        // TODO: Implement @method string getUserIdentifier()
    }

    /**
     * @return null
     */
    public function getSvslug()
    {
        return $this->svslug;
    }

    /**
     * @param null $svslug
     */
    public function setSvslug($svslug): void
    {
        $this->svslug = $svslug;
    }




    /**
     * The public representation of the user (e.g. a username, an email address, etc.)
     *
     * @see UserInterface
     */
    public function getUserIdentifier(): string
    {
        return (string) $this->username;
    }

    public function getPasswordHasherName(): ?string
    {
        // TODO: Implement getPasswordHasherName() method.
        return null;
    }
}
