<?php

namespace App\Entity\Model;

use App\Util\Inflector;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * App\Entity\Model\ItemRetencion
 *
 * @ORM\Entity(repositoryClass="App\Repository\ItemRepository")
 * @ORM\Table(indexes={
 *    @ORM\Index(name="invoice_item_desc_idx", columns={"description"})
 * })
 */
class ItemRetencion
{

    /**
     *
     * @ORM\Column(type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private  $id;

    /**
     * @var decimal $base
     *
     * @ORM\Column(type="decimal", precision=15, scale=2)
     * @Assert\NotBlank()
     */
    private $base = 0;

    /**
     * @var decimal $porcentaje
     *
     * @ORM\Column(type="decimal", precision=15, scale=2)
     */
    private $porcentaje;

    /**
     * @ORM\Column()
     */
    private  $description = null;

    /**
     * @var decimal $retenido
     *
     * @ORM\Column(type="decimal", precision=15, scale=2)
     */
    private $retenido;


    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Model\TipoRetencion")
     * @ORM\JoinColumn(name="tiporetencion_id", referencedColumnName="id", onDelete="SET NULL")
     */
    private $tipoRetencion;

    /**
     * @ORM\ManyToMany(targetEntity="App\Entity\Model\Retencion", mappedBy="items")
     */
    private $retenciones;

    /**
     * Get id
     *
     * @return integer
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Get base
     *
     * @return decimal
     */
    public function getBase()
    {
        return $this->base;
    }

    /**
     * Set Base
     *
     * @param decimal $base
     */
    public function setBase($base)
    {
        $this->base = $base;
    }

    /**
     * Get porcentaje
     *
     * @return decimal
     */
    public function getPorcentaje()
    {
        return $this->porcentaje;
    }

    /**
     * Set Porcentaje
     *
     * @param decimal $porcentaje
     */
    public function setPorcentaje($porcentaje)
    {
        $this->porcentaje = $porcentaje;
    }

    /**
     * Get description
     *
     * @return string
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * Set description
     *
     * @param string $description
     */
    public function setDescription($description)
    {
        $this->description = strtoupper($description);
    }

    /**
     * Get retenido
     *
     * @return decimal
     */
    public function getRetenido()
    {
        return $this->retenido;
    }

    /**
     * Set Retenido
     *
     * @param decimal $retenido
     */
    public function setRetenido($retenido)
    {
        $this->retenido = $retenido;
    }

    /**
     * Get tipoRetencion
     *
     * @return \App\Entity\Model\TipoRetencion
     */
    public function getTipoRetencion()
    {
        return $this->tipoRetencion;
    }

    /**
     * Set tipoRetencion
     *
     * @param \App\Entity\Model\TipoRetencion $tipoRetencion
     */
    public function setTipoRetencion(TipoRetencion $tipoRetencion)
    {
        $this->tipoRetencion = $tipoRetencion;
    }

    /** **************** CUSTOM METHODS ************* */

    /**
     * Get base amount
     *
     * @return float gross price of the item (times quantity)
     */
    public function getBaseAmount()
    {
        return $this->base;
    }

    /**
     * Get retenido amount
     *
     * @return float amount to pay
     */
    public function getGrossAmount()
    {
        return $this->retenido;
    }

    /**
     * Get porcentaje amount
     *
     * @return float amount to pay
     */
    public function getPorcentajeAmount()
    {
        return $this->porcentaje;
    }

    /**
     * Try to capture a "getTaxAmountTAXNAME" method.
     * This is to be able to use 'invoice.tax_amount_TAXNAME' in the templates
     *
     * @author JoeZ99 <jzarate@gmail.com>
     */
    public function __call($data, $argument)
    {
        if (strpos($data, 'getTaxAmount') === 0 && strlen($data)>12) {
            $tax_name = substr($data, 12);
            return $this->getTaxAmount($tax_name);
        }

        return false;
    }

    /**
     * Again, capture hipothetical {{invoice.base_amount}} and the like
     *
     */
    public function __get($name)
    {
        $method = Inflector::camelize(sprintf('get_%s', $name));
        if (method_exists($this, $method)) {
            return $this->$method();
        }

        return false;
    }

    /**
     * Twig template system needs this to answer true for the specified properties
     */
    public function __isset($name)
    {
        $classVars = array_keys(get_object_vars($this));
        $extraVars = ['base_amount', 'porcentaje_amount', 'gross_amount'];
        return in_array($name, array_merge($classVars, $extraVars));
    }

    public function __toString()
    {
        return (string) $this->description.': '.$this->quantity;
    }

    public function __construct()
    {
    }

    public function jsonSerialize()
    {
        return [
            'id' => $this->getId(),
            'retenido' => $this->getRetenido(),
            'porcentaje' => $this->getRetenido(),
            'description' => $this->getDescription(),
        ];
    }

}
