<?php

namespace App\Entity\Model;

use App\Util\Inflector;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Event\PreUpdateEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;
use function Symfony\Component\String\u;

/**
 * App\Entity\Model\GuiaRemision
 *
 * @ORM\Table(indexes={
 *    @ORM\Index(name="guiaremision_cstnm_idx", columns={"transportista_name"}),
 *    @ORM\Index(name="guiaremision_cstid_idx", columns={"transportista_identification"}),
 *    @ORM\Index(name="guiaremision_cstml_idx", columns={"transportista_email"})
 * })
 * @ORM\Entity(repositoryClass="App\Repository\GuiaRemisionRepository")
 * @ORM\HasLifecycleCallbacks()
 */
class GuiaRemision extends AbstractGuiaRemision
{
    /**
     * @ORM\Column(name="sent_by_email", type="boolean", nullable=true)
     */
    private  $sent_by_email;

    /**
     * @ORM\Column(name="number", type="integer", nullable=true)
     */
    private  $number = null;

    /**
     * @var date $issue_date
     *
     * @ORM\Column(name="issue_date", type="date", nullable=true)
     * @Assert\Type("\DateTime")
     */
    private $issue_date;

    /**
     * @var date $due_date
     *
     * @ORM\Column(name="due_date", type="date", nullable=true)
     * @Assert\Type("\DateTime")
     */
    private $due_date;

    /**
     * @ORM\ManyToMany(targetEntity="App\Entity\Model\Destinatario", cascade={"all"}, inversedBy="guiaremisions")
     * @ORM\JoinTable(name="guiaremisions_destinatarios",
     *      joinColumns={@ORM\JoinColumn(name="guiaremision_id", referencedColumnName="id", onDelete="CASCADE")},
     *      inverseJoinColumns={@ORM\JoinColumn(name="destinatario_id", referencedColumnName="id", unique=true, onDelete="CASCADE")}
     * )
     * @Assert\NotBlank(allowNull=true)
     */
    public $destinatarios;

    /**
     * @var date $fecini_transporte
     *
     * @ORM\Column(name="fecini_transporte", type="date", nullable=true)
     * @Assert\Type("\DateTime")
     */
    private $fecini_transporte;

    /**
     * @var date $fecfin_transporte
     *
     * @ORM\Column(name="fecfin_transporte", type="date", nullable=true)
     * @Assert\Type("\DateTime")
     */
    private $fecfin_transporte;

    /**
     * @var boolean $closed
     *
     * @ORM\Column(name="closed", type="boolean", nullable=true)
     */
    private $forcefully_closed = false;

    /**
     * @ORM\Column(name="claveacceso", type="string", length=49)
     */
    private  $claveAcceso = '';

    /**
     * @ORM\Column(type="string", length=6)
     */
    private  $serie = '';

    /**
     * @var text
     *
     * @ORM\Column(type="text", nullable=true)
     */
    private $xml = null;

    /**
     * @ORM\Column(type="integer", nullable=true)
     */
    private  $ambiente = null;


    /**
     * @ORM\Column( type="boolean")
     */
    private  $autorizado = false;


    /**
     * @ORM\Column(type="string", length=30,  nullable=true)
     */
    private  $fecha_autorizacion = null;

    /**
     * @ORM\Column( type="boolean")
     */
    private  $sinrespuesta = false;

    /**
     * @var text
     *
     * @ORM\Column(type="text", nullable=true)
     */
    private $mensajeError = null;

    /**
     * @var text
     *
     * @ORM\Column(type="text", nullable=true)
     */
    private $xmlAutorizado = null;

    /**
     * @ORM\Column(type="blob", nullable=true)
     */
    private $docpdf = null;

    /**
     * @ORM\Column( type="boolean")
     */
    private  $anulado = false;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Model\Empresa")
     * @ORM\JoinColumn(name="empresa_id", referencedColumnName="id", onDelete="SET NULL")
     */
    public $empresa;

    public function __construct()
    {
        parent::__construct();
        $this->issue_date = new \DateTime();
        $this->due_date = new \DateTime();
        $this->fecini_transporte = new \DateTime();

        $fecha = new \DateTime();
        $fecha->add(new \DateInterval('P5D'));

        $this->fecfin_transporte = $fecha;

    }

    /**
     * @return boolean
     */
    public function isClosed()
    {
        return $this->status === GuiaRemision::CLOSED;
    }

    /**
     * @return boolean
     */
    public function isOpen()
    {
        return in_array($this->status, [GuiaRemision::OPENED, GuiaRemision::OVERDUE], true);
    }

    /**
     * @return boolean
     */
    public function isOverdue()
    {
        return $this->status === GuiaRemision::OVERDUE;
    }

    /**
     * @return boolean
     */
    public function isDraft()
    {
        return $this->status === GuiaRemision::DRAFT;
    }

    /**
     * Set sent_by_email
     *
     * @param boolean $sentByEmail
     */
    public function setSentByEmail($sentByEmail)
    {
        $this->sent_by_email = $sentByEmail;
    }

    /**
     * Get sent_by_email
     *
     * @return boolean
     */
    public function isSentByEmail(): bool
    {
        return (bool)$this->sent_by_email;
    }

    /**
     * Set number
     *
     * @param integer $number
     */
    public function setNumber($number)
    {
        $this->number = $number;
    }

    /**
     * Get number
     *
     * @return integer
     */
    public function getNumber()
    {

        return $this->number;
    }

    /**
     * Set issue_date
     *
     * @param date $issueDate
     */
    public function setIssueDate($issueDate)
    {
        $this->issue_date = $issueDate instanceof \DateTime ?
            $issueDate : new \DateTime($issueDate);
    }

    /**
     * Get issue_date
     *
     * @return date
     */
    public function getIssueDate()
    {
        return $this->issue_date;
    }

    /**
     * Set due_date
     *
     * @param date $dueDate
     */
    public function setDueDate($dueDate)
    {
        $this->due_date = $dueDate instanceof \DateTime ?
            $dueDate : new \DateTime($dueDate);
    }

    /**
     * Get due_date
     *
     * @return date
     */
    public function getDueDate()
    {
        return $this->due_date;
    }

    public function setForcefullyClosed($value)
    {
        $this->forcefully_closed = $value;
    }

    public function isForcefullyClosed()
    {
        return $this->forcefully_closed;
    }

    /**
     * @return date
     */
    public function getFeciniTransporte()
    {
        return $this->fecini_transporte;
    }

    /**
     * @param date $fecini_transporte
     */
    public function setFeciniTransporte($fecini_transporte)
    {
        $this->fecini_transporte = $fecini_transporte;
    }

    /**
     * @return date
     */
    public function getFecfinTransporte()
    {
        return $this->fecfin_transporte;
    }

    /**
     * @param date $fecfin_transporte
     */
    public function setFecfinTransporte($fecfin_transporte)
    {
        $this->fecfin_transporte = $fecfin_transporte;
    }


    /** **************** CUSTOM METHODS AND PROPERTIES **************  */

    /**
     * TODO: provide the serie .
     */
    public function __toString()
    {
        return $this->label();
    }

    public function label(string $draftLabel = '[draft]')
    {
        /*$series = $this->getSeries();
        $label = '';
        $label .= $series ? $series->getValue() : '';
        $label .= $this->isDraft() ? $draftLabel : $this->getNumber();
*/

        $label = sprintf("%09d", $this->getNumber());
        return $label;
    }

    public function labelNumDocSustento(string $draftLabel = '[draft]')
    {
        return sprintf("%09d", $this->numeroDocSustento);
    }

    /**
     * @var int
     */
    public const DRAFT = 0;

    /**
     * @var int
     */
    public const CLOSED = 1;

    /**
     * @var int
     */
    public const OPENED = 2;

    /**
     * @var int
     */
    public const OVERDUE = 3;

    /**
     * try to catch custom methods to be used in twig templates
     */
    public function __get($name)
    {
        if (strpos($name, 'tax_amount_') === 0) {
            return $this->calculate($name, true);
        }

        //$method = Inflector::camelize("get_{$name}");
        $method = 'get'.u($name)->camel()->title()->toString();
        if (method_exists($this, $method)) {
            return $this->$method();
        }

        return false;
    }

    public function __isset($name)
    {
        if (strpos($name, 'tax_amount_') === 0) {
            return true;
        }

        if ($name == 'due_amount') {
            return true;
        }

        if (array_key_exists($name, get_object_vars($this))) {
            return true;
        }

        return parent::__isset($name);
    }

    public function getStatusString()
    {
        switch ($this->status) {
            case GuiaRemision::DRAFT;
                $status = 'draft';
                break;
            case GuiaRemision::CLOSED;
                $status = 'closed';
                break;
            case GuiaRemision::OPENED;
                $status = 'opened';
                break;
            case GuiaRemision::OVERDUE:
                $status = 'overdue';
                break;
            default:
                $status = 'unknown';
                break;
        }

        return $status;
    }

    /**
     * checkStatus
     * checks and sets the status
     *
     * @return App\GuiaRemision $this
     */
    public function checkStatus()
    {
        if ($this->status == GuiaRemision::DRAFT) {
            return $this;
        }

        if ($this->isForcefullyClosed()) {
            $this->setStatus(GuiaRemision::CLOSED);
        } elseif ($this->getDueDate()->getTimestamp() > strtotime(date('Y-m-d'))) {
            $this->setStatus(GuiaRemision::OPENED);
        } else {
            $this->setStatus(GuiaRemision::OVERDUE);
        }

        return $this;
    }

    public function checkNumber(LifecycleEventArgs $args)
    {
        // compute the number of guiaremision
        if ((!$this->number && $this->status != self::DRAFT) ||
            ($args instanceof PreUpdateEventArgs && $args->hasChangedField('series') && $this->status != self::DRAFT)
        ) {
            $repo = $args->getObjectManager()->getRepository(GuiaRemision::class);
            $series = $this->getSeries();
            if ($repo && $series)
            {
                $resp = $repo->getNextNumber($series, $this->getEmpresa()->getId());
                $this->setNumber($resp);
            }
        }
    }

    /* ********** LIFECYCLE CALLBACKS *********** */

    /**
     * @ORM\PrePersist
     * @ORM\PreUpdate
     */
    public function preSave(LifecycleEventArgs $args)
    {
        $this->setUpdatedAt(new \DateTime('now'));

        if ($this->getCreatedAt() == null) {
            $this->setCreatedAt(new \DateTime('now'));
        }

        parent::presave($args);

        $this->checkNumber($args);

    }

    /**
     * @return string
     */
    public function getCustomermicro()
    {
        return $this->transportistamicro;
    }

    /**
     * @param string $transportistamicro
     */
    public function setCustomermicro(string $transportistamicro)
    {
        $this->transportistamicro = $transportistamicro;
    }

    /**
     * @return string
     */
    public function getClaveAcceso()
    {
        return $this->claveAcceso;
    }

    /**
     * @param string $claveAcceso
     */
    public function setClaveAcceso($claveAcceso)
    {
        $this->claveAcceso = $claveAcceso;
    }

    /**
     * @return string
     */
    public function getSerie()
    {
        return $this->serie;
    }

    /**
     * @param string $serie
     */
    public function setSerie($serie)
    {
        $this->serie = $serie;
    }

    /**
     * @return bool
     */
    public function getAutorizado()
    {
        return $this->autorizado;
    }

    /**
     * @param bool $autorizado
     */
    public function setAutorizado($autorizado)
    {
        $this->autorizado = $autorizado;
    }

    /**
     * @return string
     */
    public function getAutorizacion()
    {
        return $this->autorizacion;
    }

    /**
     * @param string $autorizacion
     */
    public function setAutorizacion($autorizacion)
    {
        $this->autorizacion = $autorizacion;
    }


    /**
     * @return text
     */
    public function getXml()
    {
        return $this->xml;
    }

    /**
     * @param text $xml
     */
    public function setXml($xml)
    {
        $this->xml = $xml;
    }

    /**
     * @return mixed
     */
    public function getEmpresa()
    {
        return $this->empresa;
    }

    /**
     * @param mixed $empresa
     */
    public function setEmpresa($empresa)
    {
        $this->empresa = $empresa;
    }

    /**
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getDestinatarios()
    {
        return $this->destinatarios;
    }

    /**
     * @param mixed $destinatarios
     */
    public function setDestinatarios($destinatarios)
    {
        $this->destinatarios = $destinatarios;
    }

    /**
     * @return string
     */
    public function getFechaAutorizacion()
    {
        return $this->fecha_autorizacion;
    }

    /**
     * @param string $fecha_autorizacion
     */
    public function setFechaAutorizacion(string $fecha_autorizacion)
    {
        $this->fecha_autorizacion = $fecha_autorizacion;
    }

    /**
     * @return int
     */
    public function getAmbiente()
    {
        return $this->ambiente;
    }

    /**
     * @param int $ambiente
     */
    public function setAmbiente($ambiente)
    {
        $this->ambiente = $ambiente;
    }

    /**
     * @param mixed $ambiente
     */
    public function getLabelAmbiente()
    {
        if ($this->ambiente === 1) {
            return "PRUEBAS";
        } elseif ($this->ambiente === 2) {
            return "PRODUCCION";
        } else
            return "UNDEFINED";

    }

    /**
     * @return bool
     */
    public function isSinrespuesta()
    {
        return $this->sinrespuesta;
    }

    /**
     * @param bool $sinrespuesta
     */
    public function setSinrespuesta($sinrespuesta)
    {
        $this->sinrespuesta = $sinrespuesta;
    }

    /**
     * @return text
     */
    public function getMensajeError()
    {
        return $this->mensajeError;
    }

    /**
     * @param text $mensajeError
     */
    public function setMensajeError($mensajeError)
    {
        $this->mensajeError = $mensajeError;
    }

    /**
     * @return text
     */
    public function getXmlAutorizado()
    {
        return $this->xmlAutorizado;
    }

    /**
     * @param text $xmlAutorizado
     */
    public function setXmlAutorizado($xmlAutorizado)
    {
        $this->xmlAutorizado = $xmlAutorizado;
    }

    /**
     * @return mixed
     */
    public function getDocpdf()
    {
        return $this->docpdf;
    }

    /**
     * @param mixed $docpdf
     */
    public function setDocpdf($docpdf)
    {
        $this->docpdf = $docpdf;
    }

    /**
     * @return bool
     */
    public function getAnulado(): bool
    {
        return $this->anulado;
    }

    /**
     * @param bool $anulado
     */
    public function setAnulado(bool $anulado): void
    {
        $this->anulado = $anulado;
    }

    /**
     * @return string
     */
    public function getEstado(): string
    {
        if ($this->autorizado)
            return 'A';
        else
            return 'N';
    }
}
