<?php

namespace App\Entity\Model;

use App\Util\Inflector;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\Common\Collections\ArrayCollection;
use Gedmo\Timestampable\Traits\TimestampableEntity;
use Gedmo\Timestampable\Timestampable;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * App\Entity\Model\AbstractLiquidacion
 *
 * @ORM\MappedSuperclass
 * @ORM\HasLifecycleCallbacks
 */
abstract class AbstractLiquidacion implements Timestampable
{
    /**
     * @var \Doctrine\Common\Collections\ArrayCollection|\App\Entity\Model\Item[]|mixed|mixed[]
     */
    public $items;

    /**
     * @var \Doctrine\Common\Collections\ArrayCollection|\App\Entity\Model\ItemPago[]|mixed|mixed[]
     */
    public $pagos;

    /**
     * @var \Doctrine\Common\Collections\ArrayCollection|\App\Entity\Model\Reembolso[]|mixed|mixed[]|null
     */
    public $reembolsos;

    public $empresa;

    public $esReembolso;

    use TimestampableEntity;

    /**
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private  $id = null;

    /**
     * @ORM\Column(name="slug", type="string", length=32, nullable=true, unique=true)
     */
    private  $slug = null;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Model\Provider", inversedBy="liquidacions")
     * @ORM\JoinColumn(name="provider_id", referencedColumnName="id")
     */
    protected $provider;

    /**
     *
     * @ORM\Column(name="provider_name", type="string", length=255)
     * @Assert\NotBlank(message = "Ingrese el nombre del cliente")
     */
    private  $provider_name = null;

    /**
     *
     * @ORM\Column(name="provider_identification", type="string", length=128, nullable=true)
     * @Assert\NotBlank(message = "Ingrese el número de identificación del cliente")
     */
    private  $provider_identification = null;

    /**
     *
     * @ORM\Column(name="provider_email", type="string", length=255)
     * @Assert\NotBlank(message = "Ingrese el correo electrónico del cliente")
     * @Assert\Email()
     */
    private  $provider_email = null;

    /**
     * @var text $provider_address
     *
     * @ORM\Column(name="provider_address", type="text", nullable=true)
     */
    private $provider_address = null;

    /**
     * @var text $notes
     *
     * @ORM\Column(name="notes", type="text", nullable=true)
     */
    private $notes = null;

    /**
     * @var decimal $base_amount
     *
     * @ORM\Column(name="base_amount", type="decimal", scale=2, precision=15, nullable=true)
     */
    private $base_amount = null;

    /**
     * @var decimal $discount_amount
     *
     * @ORM\Column(name="discount_amount", type="decimal", scale=2, precision=15, nullable=true)
     */
    private $discount_amount = null;

    /**
     * @var decimal $net_amount
     *
     * @ORM\Column(name="net_amount", type="decimal", scale=2, precision=15, nullable=true)
     */
    private $net_amount = null;

    /**
     * @var decimal $gross_amount
     *
     * @ORM\Column(name="gross_amount", type="decimal", scale=2, precision=15, nullable=true)
     */
    private $gross_amount = null;

    /**
     * @var decimal $paid_amount
     *
     * @ORM\Column(name="paid_amount", type="decimal", scale=2, precision=15, nullable=true)
     */
    private $paid_amount = null;

    /**
     * @var decimal $tax_amount
     *
     * @ORM\Column(name="tax_amount", type="decimal", scale=2, precision=15, nullable=true)
     */
    private $tax_amount = null;

    /**
     * @var smallint $status
     *
     * @ORM\Column(name="status", type="smallint", nullable=true)
     * @Assert\Length(min=0, max=3)
     */
    protected $status = 0;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Model\Series")
     * @ORM\JoinColumn(name="series_id", referencedColumnName="id")
     * @Assert\NotBlank()
     *
     * unidirectional many-to-one
     */
    private $series;

    /**
     * @ORM\Column(type="integer", nullable=true)
     */
    private  $usuario = null;

    /**
     * @var decimal $base_reembo_amount
     *
     * @ORM\Column(name="base_reembo_amount", type="decimal", scale=2, precision=15, nullable=true)
     */
    private $base_reembo_amount = null;

    /**
     * @var decimal $tax_reembo_amount
     *
     * @ORM\Column(name="tax_reembo_amount", type="decimal", scale=2, precision=15, nullable=true)
     */
    private $tax_reembo_amount = null;

    /**
     * @var decimal $gross_reembo_amount
     *
     * @ORM\Column(name="gross_reembo_amount", type="decimal", scale=2, precision=15, nullable=true)
     */
    private $gross_reembo_amount = null;

    public function __construct()
    {
        $this->items = new ArrayCollection();
        $this->pagos = new ArrayCollection();
        $this->reembolsos = new ArrayCollection();
    }

    /**
     * Get id
     *
     * @return integer
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set provider parameters from provider entity.
     *
     * @param App\Entity\Model\Provider $provider
     */
    public function setFromProvider(Provider $provider)
    {
        if (empty($this->provider_name) && $provider->getName()) {
            $this->provider_name = $provider->getName();
        }

        if (empty($this->provider_email) && $provider->getEmail()) {
            $this->provider_email = $provider->getEmail();
        }

        if (empty($this->provider_identification) && $provider->getIdentification()) {
            $this->provider_identification = $provider->getIdentification();
        }

        if (empty($this->provider_address) && $provider->getAddress()) {
            $this->provider_address = $provider->getAddress();
        }

    }

    /**
     * Set provider.
     *
     * @param App\Entity\Model\Provider $provider
     */
    public function setProvider(Provider $provider)
    {
        $this->provider = $provider;
    }

    /**
     * @return mixed
     */
    public function getProvider()
    {
        return $this->provider;
    }

    /**
     * Set provider_name
     *
     * @param string $providerName
     */
    public function setProviderName($providerName)
    {
        $this->provider_name = $providerName;
    }

    /**
     * Get provider_name
     *
     * @return string
     */
    public function getProviderName()
    {
        return $this->provider_name;
    }

    /**
     * Set provider_identification
     *
     * @param string $providerIdentification
     */
    public function setProviderIdentification($providerIdentification)
    {
        $this->provider_identification = $providerIdentification;
    }

    /**
     * Get provider_identification
     *
     * @return string
     */
    public function getProviderIdentification()
    {
        return $this->provider_identification;
    }

    /**
     * Set provider_email
     *
     * @param string $providerEmail
     */
    public function setProviderEmail($providerEmail)
    {
        $this->provider_email = $providerEmail;
    }

    /**
     * Get provider_email
     *
     * @return string
     */
    public function getProviderEmail()
    {
        return $this->provider_email;
    }

    /**
     * Set provider_address
     *
     * @param text $providerAddress
     */
    public function setProviderAddress($providerAddress)
    {
        $this->provider_address = $providerAddress;
    }

    /**
     * Get provider_address
     *
     * @return text
     */
    public function getProviderAddress()
    {
        return $this->provider_address;
    }

    /**
     * Set notes
     *
     * @param text $notes
     */
    public function setNotes($notes)
    {
        $this->notes = strtoupper($notes);
    }

    /**
     * Get notes
     *
     * @return text
     */
    public function getNotes()
    {
        return $this->notes;
    }

    /**
     * Set base_amount
     *
     * @param decimal $baseAmount
     */
    public function setBaseAmount($baseAmount)
    {
        $this->base_amount = $baseAmount;
    }

    /**
     * Get base_amount
     *
     * @return decimal
     */
    public function getBaseAmount()
    {
        return $this->base_amount;
    }

    /**
     * Set discount_amount
     *
     * @param decimal $discountAmount
     */
    public function setDiscountAmount($discountAmount)
    {
        $this->discount_amount = $discountAmount;
    }

    /**
     * Get discount_amount
     *
     * @return decimal
     */
    public function getDiscountAmount()
    {
        return $this->discount_amount;
    }

    /**
     * Set net_amount
     *
     * @param decimal $netAmount
     */
    public function setNetAmount($netAmount)
    {
        $this->net_amount = $netAmount;
    }

    /**
     * Get net_amount
     *
     * @return decimal
     */
    public function getNetAmount()
    {
        return $this->net_amount;
    }

    /**
     * Set gross_amount
     *
     * @param decimal $grossAmount
     */
    public function setGrossAmount($grossAmount)
    {
        $this->gross_amount = $grossAmount;
    }

    /**
     * Get gross_amount
     *
     * @return decimal
     */
    public function getGrossAmount()
    {
        return $this->gross_amount;
    }

    /**
     * Set paid_amount
     *
     * @param decimal $paidAmount
     */
    public function setPaidAmount($paidAmount)
    {
        $this->paid_amount = $paidAmount;
    }

    /**
     * Get paid_amount
     *
     * @return decimal
     */
    public function getPaidAmount()
    {
        return $this->paid_amount;
    }

    /**
     * Set tax_amount
     *
     * @param decimal $taxAmount
     */
    public function setTaxAmount($taxAmount)
    {
        $this->tax_amount = $taxAmount;
    }

    /**
     * Get tax_amount
     *
     * @return decimal
     */
    public function getTaxAmount()
    {
        return $this->tax_amount;
    }

    /**
     * Set status
     *
     * @param integer $status
     */
    public function setStatus($status)
    {
        $this->status = $status;
    }

    /**
     * Get status
     *
     * @return integer
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * Set serie
     *
     * @param App\Entity\Model\Series $series
     */
    public function setSeries(Series $series)
    {
        $this->series = $series;
    }

    /**
     * Get serie
     *
     * @return App\Entity\Model\Series
     */
    public function getSeries()
    {
        return $this->series;
    }

    /**
     * Add items
     *
     * @param App\Entity\Model\Item $item
     */
    public function addItem(Item $item)
    {
        $this->items[] = $item;
    }

    /**
     * Get items
     *
     * @return Doctrine\Common\Collections\Collection
     */
    public function getItems()
    {
        return $this->items;
    }


    /**
     * Add pagos
     *
     * @param App\Entity\Model\ItemPago $item
     */
    public function addPago(ItemPago $item)
    {
        $this->pagos[] = $item;
    }

    /**
     * Get pagos
     *
     * @return Doctrine\Common\Collections\Collection
     */
    public function getPagos()
    {
        return $this->pagos;
    }

    /**
     * Add reembolso
     *
     * @param App\Entity\Model\ItemReembolso $item
     */
    public function addReembolso(Reembolso $item)
    {
        $this->reembolsos[] = $item;
    }

    /**
     * Get pagos
     *
     * @return Doctrine\Common\Collections\Collection
     */
    public function getReembolsos()
    {
        return $this->reembolsos;
    }

    /**
     * @return decimal
     */
    public function getBaseReemboAmount()
    {
        return $this->base_reembo_amount;
    }

    /**
     * @param decimal $base_reembo_amount
     */
    public function setBaseReemboAmount($base_reembo_amount): void
    {
        $this->base_reembo_amount = $base_reembo_amount;
    }

    /**
     * @return decimal
     */
    public function getTaxReemboAmount()
    {
        return $this->tax_reembo_amount;
    }

    /**
     * @param decimal $tax_reembo_amount
     */
    public function setTaxReemboAmount($tax_reembo_amount): void
    {
        $this->tax_reembo_amount = $tax_reembo_amount;
    }

    /**
     * @return decimal
     */
    public function getGrossReemboAmount()
    {
        return $this->gross_reembo_amount;
    }

    /**
     * @param decimal $gross_reembo_amount
     */
    public function setGrossReemboAmount($gross_reembo_amount): void
    {
        $this->gross_reembo_amount = $gross_reembo_amount;
    }


    /** ########### CUSTOM METHODS ################## */


    public function __isset($name)
    {
        return array_key_exists($name, get_object_vars($this));
    }

    /** ** RELATIONSHIPS ** */


    /**
     * removeItem
     * removes an item and recalculcates amounts
     *
     * @param mixed $mixed : can be an integer or an item instance
     *                       - if an integer, removes the item with
     *                         that position in the collection
     *                       - if an instance, removes that item
     * @author JoeZ99 <jzarate@gmail.com>
     */
    public function removeItem($mixed)
    {
        if ($mixed instanceof Item) {
            $items = $this->getItems();
            foreach ($items as $ref => $item) {
                if ($item === $mixed) {
                    unset($items[$ref]);
                    break;
                }
            }
        } elseif (is_int($mixed)) {
            unset($this->items[$mixed]);
        }

    }

    public function removePago($mixed)
    {
        if ($mixed instanceof ItemPago) {
            $pagos = $this->getPagos();
            foreach ($pagos as $ref => $pago) {
                if ($pago === $mixed) {
                    unset($pagos[$ref]);
                    break;
                }
            }
        } elseif (is_int($mixed)) {
            unset($this->pagos[$mixed]);
        }

    }

    public function removeReembolso($mixed)
    {
        if ($mixed instanceof Reembolso) {
            $reembolsos = $this->getReembolsos();
            foreach ($reembolsos as $ref => $reembolso) {
                if ($reembolso === $mixed) {
                    unset($reembolsos[$ref]);
                    break;
                }
            }
        } elseif (is_int($mixed)) {
            unset($this->reembolsos[$mixed]);
        }

    }

    /* ** OTHER ** */

    private $decimals = null;

    public function getRoundedAmount($concept = 'gross')
    {
        if (!in_array($concept, ['base', 'discount', 'net', 'tax', 'gross'])) {
            return 0;
        }

        return round(call_user_func([$this, Inflector::camelize('get_'.$concept.'_amount')]), $this->getDecimals());
    }

    private function getDecimals()
    {
        if (!$this->decimals) {
            $this->decimals = 2;
        }

        return $this->decimals;
    }


    /**
     * calculate values over items
     *
     * Warning!! this method only works when called from a real entity, not
     * the abstract.
     *
     * @param string $field
     * @param boolean $rounded
     * @return float
     */
    public function calculate($field, $rounded = false)
    {
        $val = 0;
        switch ($field) {
            case 'paid_amount':
                /*foreach ($this->getPayments() as $payment) {
                    $val += $payment->getAmount();
                }*/
                break;
            default:
                foreach ($this->getItems() as $item) {
                    $method = Inflector::camelize('get_'.$field);
                    $val += $item->$method();
                }
                break;
        }

        if ($rounded) {
            return round($val, $this->getDecimals());
        }

        return $val;
    }

    /**
     * calculate values over items
     *
     * Warning!! this method only works when called from a real entity, not
     * the abstract.
     *
     * @param string $field
     * @param boolean $rounded
     * @return float
     */
    public function calculateReembo($field, $rounded = false)
    {
        $val = 0;
        switch ($field) {
        }

        if ($rounded) {
            return round($val, $this->getDecimals());
        }

        return $val;
    }

    public function checkAmounts()
    {
        $this->setBaseAmount($this->calculate('base_amount'));
        $this->setDiscountAmount($this->calculate('discount_amount'));
        $this->setNetAmount($this->getBaseAmount() - $this->getDiscountAmount());
        $this->setTaxAmount($this->calculate('tax_amount'));
        $rounded_gross = round(
            $this->getNetAmount() + $this->getTaxAmount(),
            $this->getDecimals()
        );
        $this->setGrossAmount($rounded_gross);

        return $this;
    }

    public function checkReemboAmounts()
    {
        $this->setBaseReemboAmount($this->calculateReembo('base_amount'));
        //$this->setDiscountAmount($this->calculate('discountreembo__amount'));
        //$this->setNetAmount($this->getBaseAmount() - $this->getDiscountAmount());
        $this->setTaxReemboAmount($this->calculateReembo('tax_amount'));
        $rounded_gross = round(
            $this->getBaseReemboAmount() + $this->getTaxReemboAmount(),
            $this->getDecimals()
        );
        $this->setGrossReemboAmount($rounded_gross);

        return $this;
    }


    /** *********** LIFECYCLE CALLBACKS ************* */

    /**
     * @ORM\PreUpdate
     * @ORM\PrePersist
     */
    public function preSave(LifecycleEventArgs $args)
    {
        $this->checkStatus();
        $providers = $args->getObjectManager()->getRepository(Provider::class)->findBy([
            'name' => $this->getProviderName(),
            'identification' => $this->getProviderIdentification(),
            'empresa' => $this->getEmpresa(),
        ]);
        $provider = reset($providers);
        if ($provider) {
            $this->setProvider($provider);
        }

        if($this->id === null)
            $this->slug = md5('sv_'.$this->empresa->getId().uniqid());

        if($this->esReembolso === false){
            $this->reembolsos = null;
        }

    }

    /**
     * @return mixed
     */
    public function getUsuario()
    {
        return $this->usuario;
    }

    /**
     * @param mixed $usuario
     */
    public function setUsuario($usuario)
    {
        $this->usuario = $usuario;
    }

    /**
     * @return string
     */
    public function getSlug(): string
    {
        return $this->slug;
    }


}
