<?php

namespace App\Controller;

use App\Service\EmFactory;
use App\Util\Funciones;
use App\Util\RestApiFunciones;
use Knp\Component\Pager\PaginatorInterface;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Symfony\Component\Form\Extension\Core\Type\FileType;
use Symfony\Component\HttpFoundation\File\Exception\FileException;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use App\Entity\Model\Provider;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

/**
 * @Route("/provider")
 */
class ProviderController extends AbstractController
{
    protected $translator;

    /**
     * @Route("", name="provider_index")
     *
     */
    public function indexAction(EmFactory $emFactory, Request $request, PaginatorInterface $paginator, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $repo = $em->getRepository(\App\Entity\Model\Provider::class);
        $repo->setPaginator($paginator);
        // @todo Unhardcode this.
        $limit = 50;

        $form = $this->createForm(\App\Form\SearchProviderType::class, null, [
            'action' => $this->generateUrl('provider_index'),
            'method' => 'GET',
        ]);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $pagination = $repo->paginatedSearch($form->getData(), $limit, $request->query->getInt('page', 1), $empresa->getId());
        } else {
            $pagination = $repo->paginatedSearch([], $limit, $request->query->getInt('page', 1), $empresa->getId());
        }

        $providers = [];
        foreach ($pagination->getItems() as $item) {
            $providers[] = $item;
        }

        $listForm = $this->createForm(\App\Form\ListGenericType::class, $providers, [
            'action' => $this->generateUrl('provider_index'),
        ]);
        $listForm->handleRequest($request);
        if ($listForm->isSubmitted() && $listForm->isValid()) {
            $data = $listForm->getData();
            if ($request->request->has('delete')) {
                if (empty($data['providers'])) {
                    $this->addTranslatedMessage('flash.nothing_selected', 'warning');
                } else {
                    foreach ($data['providers'] as $provider) {
                        $em->remove($provider);
                    }

                    $em->flush();
                    $this->addTranslatedMessage('flash.bulk_deleted');

                    // Rebuild the query, since some objects are now missing.
                    return $this->redirectToRoute('provider_index');
                }
            }
        }

        return $this->render('Provider\index.html.twig',
            [
                'providers' => $pagination,
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $empresa == null ? 'USD' : $empresa->getCurrency(),
                'search_form' => $form->createView(),
                'list_form' => $listForm->createView(),
            ]);
    }

    /**
     * @Route("/autocomplete", name="provider_autocomplete")
     */
    public function autocompleteAction(EmFactory $emFactory, Request $request): \Symfony\Component\HttpFoundation\Response
    {

        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $entities = $em
            ->getRepository(\App\Entity\Model\Provider::class)
            ->findLike($request->get('term'), $empresa->getId());

        return new JsonResponse($entities);
    }

    /**
     * @Route("/add", name="provider_add")
     *
     */
    public function addAction(EmFactory $emFactory, Request $request, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $provider = new Provider();
        $provider->setEmpresa($empresa);

        $form = $this->createForm(\App\Form\ProviderType::class, $provider, [
            'action' => $this->generateUrl('provider_add'),
        ]);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            $numDocumento = $provider->getIdentification();
            //$respuesta = $provider->validarRuc($numDocumento, $provider->getTipoDocumento());

            $respuesta = null;
            if ($respuesta === null) {

                $entities = $em->getRepository(\App\Entity\Model\Provider::class)->findIdentificacion($numDocumento, $empresa);

                if ((is_countable($entities) ? count($entities) : 0) > 0) {
                    $this->addTranslatedMessage('Número de identificación ya se encuentra registrado', 'danger');
                } else {
                    $em->persist($provider);
                    $em->flush();

                    $this->addTranslatedMessage('Proveedor guardado');
                    return $this->redirectToRoute('provider_index');
                }
            } else {
                $this->addTranslatedMessage('Documento del Proveedor NO Válido ', 'danger');
                //return $this->redirectToRoute('registro_nuevo');
            }
        }

        return $this->render('Provider\edit.html.twig',
            ['form' => $form->createView(), 'entity' => $provider]);
    }

    /**
     * @Route("/edit/{slug}prv{id}", name="provider_edit")
     *
     */
    public function editAction($id, EmFactory $emFactory, Request $request, $slug, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $provider = $em->getRepository(\App\Entity\Model\Provider::class)->findBySlug($slug, $id);
        if (!$provider) {
            throw $this->createNotFoundException('Unable to find Provider entity.');
        }

        $form = $this->createForm(\App\Form\ProviderType::class, $provider, [
            'action' => $this->generateUrl('provider_edit', ['slug' => $slug, 'id' => $id]),
        ]);
        $form->handleRequest($request);

        if ($form->isSubmitted()) {
            if ($request->request->has('Form-delete')) {
                if ($provider->getIdentification() === '9999999999999') {
                    $this->addTranslatedMessage('Consumidor Final no puede ser eliminado', 'warning');
                } else {
                    $em->remove($provider);
                    $em->flush();
                    $this->addTranslatedMessage('Proveedor Eliminado');
                }

                return $this->redirectToRoute('provider_index');
            }

            if ($form->isValid()) {
                $numDocumento = $provider->getIdentification();

                $respuesta = null;

                //if ($provider->getIdentification() != '9999999999999')
                //    $respuesta = $provider->validarRuc($numDocumento, $provider->getTipoDocumento());
                //else
                //    $provider->setTipoDocumento('*');

                if ($respuesta === null) {
                    $provider->setEmpresa($empresa);
                    $em->persist($provider);
                    $em->flush();
                    $this->addTranslatedMessage('Proveedor Actualizado');

                    return $this->redirectToRoute('provider_edit', ['slug' => $slug, 'id' => $id]);

                } else {
                    $this->addTranslatedMessage('Documento del Proveedor NO Válido ', 'danger');
                    //return $this->redirectToRoute('registro_nuevo');
                }
            }
        }

        return $this->render('Provider\edit.html.twig',
            ['form' => $form->createView(), 'entity' => $provider]);
    }

    protected function addTranslatedMessage($message, $status = 'success')
    {
        $this->addFlash($status, $this->translator->trans($message, [], 'provider'));

    }


    /**
     * @Route("/searchprovider", name="provider_searchprovider")
     */
    public function searchProveedorAction(Request $request, EmFactory $emFactory): \Symfony\Component\HttpFoundation\Response
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $entities = $em
            ->getRepository(\App\Entity\Model\Provider::class)
            ->findIdentificacion($request->get('term'), $empresa);

        if (count($entities) > 0) {

            if (count($entities) > 1) {
                return new JsonResponse('ERROR, Se encontraron mas de un registro con el número de identificación ingresado',
                    \Symfony\Component\HttpFoundation\Response::HTTP_BAD_REQUEST);
            } else {
                return new JsonResponse(['message' => 'Proveedor', 'item' => $entities[0]->jsonSerialize()], \Symfony\Component\HttpFoundation\Response::HTTP_OK);
            }

        } else {
            return new JsonResponse(
                'ERROR, Proveedor no existe'
                , \Symfony\Component\HttpFoundation\Response::HTTP_BAD_REQUEST);
        }

    }

    /**
     * @Route("/rest/add", name="rest_provider_add")
     */
    public function addProviderAction(Request $request, EmFactory $emFactory, ValidatorInterface $validator): \Symfony\Component\HttpFoundation\Response
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $provider = new Provider();

        $form = $this->createForm(\App\Form\ProviderType::class, $provider, [
            'action' => $this->generateUrl('provider_add'),
        ]);
        $form->handleRequest($request);

        if ($form->isSubmitted()) {
            $provider->setEmpresa($empresa);
            $errors = $validator->validate($provider);
            if (count($errors) > 0)
                return new JsonResponse(RestApiFunciones::getErrorValidacion($errors), \Symfony\Component\HttpFoundation\Response::HTTP_UNAUTHORIZED);
            else {
                $numDocumento = $provider->getIdentification();
                //$respuesta = $provider->validarRuc($numDocumento, $provider->getTipoDocumento());

                $respuesta = null;
                if ($respuesta === null) {
                    $entities = $em->getRepository(\App\Entity\Model\Provider::class)->findIdentificacion($numDocumento, $empresa);

                    if ((is_countable($entities) ? count($entities) : 0) > 0) {
                        return new JsonResponse('ERROR, YA SE ENCUENTRA REGISTRADO EL NUMERO DE DOCUMENTO: ' . $numDocumento,
                            \Symfony\Component\HttpFoundation\Response::HTTP_BAD_REQUEST);
                    }

                    try {
                        $em->persist($provider);
                        $em->flush();

                return new JsonResponse(['message' => 'Proveedor', 'item' => $provider->jsonSerialize()], \Symfony\Component\HttpFoundation\Response::HTTP_OK);
                    } catch (\Symfony\Component\Config\Definition\Exception\Exception $exception) {
                        return new JsonResponse('ERROR, ' . $exception->getMessage(),
                            \Symfony\Component\HttpFoundation\Response::HTTP_BAD_REQUEST);
                    }

                } else {
                    return new JsonResponse('ERROR, Documento del Proveedor NO Válido ',
                        \Symfony\Component\HttpFoundation\Response::HTTP_BAD_REQUEST);
                }
            }
        }

        return new JsonResponse('ERROR, Desconocido consulte con el administrador ',
            \Symfony\Component\HttpFoundation\Response::HTTP_BAD_REQUEST);
    }

    /**
     * @Route("/search-reference", name="provider_search_reference")
     */
    public function searchReferenceAction(EmFactory $emFactory, Request $request): \Symfony\Component\HttpFoundation\Response
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');

        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $limit = 5;

        $aux = $request->query->all();

        $page = 0;

        $start = $page > 1 ? ($page - 1) * $limit : 0;

        $condition = '';

        if (isset($aux['sEcho'])) {
            $sEcho = $aux['sEcho'];
        }
        if (isset($aux['iDisplayStart'])) {
            $start = intval($aux['iDisplayStart']);
        }
        if (isset($aux['iDisplayLength'])) {
            $iDisplayLength = intval($aux['iDisplayLength']);
        }
        if (isset($aux['sSearch'])) {
            $condition = Funciones::cleanParametro($aux['sSearch']);
        }

        $repo = $em->getRepository(Provider::class);

        $sql = "";
        $paginator = $repo->paginatedAjax($sql, $condition, $limit, $start, $empresa->getId());

        $total_data = is_countable($paginator) ? count($paginator) : 0;

        $providerArray = [];
        //$i = 0;
        foreach ($paginator as $entity) {
            //$entity = new Provider();
            $providerArray[] = [
                $entity->getSlug(),
                $entity->getIdentification(),
                $entity->getName(),
                $entity->getEmail(),
                $entity->getAddress()
            ];
            //$i++;
        }

        $arr = array(
            "iTotalRecords" => $total_data,
            "iTotalDisplayRecords" => $total_data,
            'aaData' => $providerArray
        );

        $post_data = json_encode($arr);

        return new Response($post_data, 200, array('Content-Type' => 'application/json'));

    }

    /**
     * @Route("desplantilla", name="provider_plantilla_download")
     *
     */
    public function plantilladownloadAction(EmFactory $emFactory, Request $request, TranslatorInterface $translator, PaginatorInterface $paginator)
    {

        $format = 'xls';
        $filename = 'proveedores' . '.' . $format;

        $contentType = 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet';

        $content = file_get_contents($filename);
        $response = new Response();
        $response->headers->set('Content-Type', $contentType);
        $response->headers->set('Content-Disposition', 'attachment;filename="' . $filename . '"');

        $response->headers->addCacheControlDirective('no-cache', true);
        $response->headers->addCacheControlDirective('must-revalidate', true);

        $response->setContent($content);
        return $response;
    }

    /**
     * @Route("upload", name="provider_upload")
     *
     */
    public function uploadAction(EmFactory $emFactory, Request $request, TranslatorInterface $translator, PaginatorInterface $paginator)
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository('App\Entity\Model\Empresa');
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $data["fileprovider"] = '';

        $form = $this->createFormBuilder($data)
            ->add('fileprovider', FileType::class, [
                'required' => true,
                //'attr' => ['style' =>"color: transparent"],
                'label' => 'Archivo Proveedores',
            ])
            ->setAction($this->generateUrl('provider_upload'))
            ->getForm();

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $formData = $form->getData();

            if (!empty($formData['fileprovider'])) {
                /** @var Symfony\Component\HttpFoundation\File\UploadedFile $file */
                $file = $formData['fileprovider'];

                $ext = $file->getClientOriginalName();
                $ext = explode('.', $ext);
                if (strtolower($ext[1]) === 'xls' || strtolower($ext[1]) === 'xls') {

                    // Move the file to the uploads directory.
                    $uploadsDir = "temp/";

                    if (!file_exists($uploadsDir)) {
                        mkdir($uploadsDir, 0755, true);
                    }

                    $fileName = $empresa->getRuc() . '.provider.xls';

                    try {
                        $newFile = $file->move($uploadsDir, $fileName);

                        $spreadsheet = IOFactory::load($uploadsDir . $fileName); // Here we are able to read from the excel file
                        $row = $spreadsheet->getActiveSheet()->removeRow(1); // I added this to be able to remove the first file line
                        $sheetData = $spreadsheet->getActiveSheet()->toArray(null, true, true, true); // here, the read data is turned into an array
                        $añadidos = 0;
                        $repetidos =0;
                        foreach ($sheetData as $Row) {

                            $nombre = $Row['A']; // store the first_name on each iteration
                            $ruc = $Row['B']; // store the last_name on each iteration
                            $email = $Row['C'] === null ? "" : $Row['C'];     // store the email on each iteration
                            $direccion = $Row['D'];
                            $tpdocumento = $Row['E'];
                            $telefono = $Row['F'];
                            $tppersona = $Row['G'];

                            $repo = $em->getRepository(Provider::class);

                            //$cliente = $repo->findOneBy(array('reference' => $codigo));
                            //if (!$producto) {
                            $cliente = new Provider();
                            $cliente->setName($nombre);
                            $cliente->setIdentification($ruc);
                            $cliente->setEmail($email);
                            $cliente->setAddress($direccion);
                            $cliente->setTipoDocumento($tpdocumento);
                            $cliente->setTelefono($telefono);

                            $cliente->setTipoProveedor($tppersona);
                            $cliente->setEmpresa($empresa);
                            $em->persist($cliente);

                            $añadidos++;
                            // here Doctrine checks all the fields of all fetched data and make a transaction to the database.
                            //}
                            //else
                            //    $repetidos++;
                        }
                        $em->flush();
                        $em->clear();

                    } catch (FileException $e) {
                        $this->addTranslatedMessage('ERROR CARGAR ARCHIVO', 'danger');
                    }

                    $this->addTranslatedMessage('Proveedores insertados: '. $añadidos. PHP_EOL);
                    return $this->redirect($this->generateUrl('provider_index'));
                } else {
                    $this->addTranslatedMessage('Extension archivo no valido', 'warning');
                }

            }
        }

        return $this->render('Provider\upload.html.twig',
            array(
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'form' => $form->createView(),
            ));
    }

}
