<?php

namespace App\Controller;

use App\Entity\Model\Product;
use App\Entity\Model\Tax;
use App\Service\EmFactory;
use App\Util\Funciones;
use Exception;
use Knp\Component\Pager\PaginatorInterface;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Form\Extension\Core\Type\FileType;
use Symfony\Component\HttpFoundation\File\Exception\FileException;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Contracts\Translation\TranslatorInterface;

/**
 * @Route("/product")
 */
class ProductController extends AbstractController
{
    protected $translator;

    /**
     * @Route("", name="product_index")
     *
     */
    public function indexAction(EmFactory $emFactory, Request $request, TranslatorInterface $translator, PaginatorInterface $paginator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $repo = $em->getRepository(\App\Entity\Model\Product::class);
        $repo->setPaginator($paginator);
        // @todo Unhardcode this.
        $limit = 50;

        $form = $this->createForm(\App\Form\SearchProductType::class, null, [
            'action' => $this->generateUrl('product_index'),
            'method' => 'GET',
        ]);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $pagination = $repo->paginatedSearch($form->getData(), $limit, $request->query->getInt('page', 1), $empresa->getId());
        } else {
            $pagination = $repo->paginatedSearch([], $limit, $request->query->getInt('page', 1), $empresa->getId());
        }

        $products = [];
        foreach ($pagination->getItems() as $item) {
            $products[] = $item[0];
        }

        $listForm = $this->createForm(\App\Form\ListGenericType::class, $products, [
            'action' => $this->generateUrl('product_index'),
        ]);
        $listForm->handleRequest($request);
        if ($listForm->isSubmitted() && $listForm->isValid()) {
            $data = $listForm->getData();
            if ($request->request->has('delete')) {
                if (empty($data['products'])) {
                    $this->addTranslatedMessage('flash.nothing_selected', 'warning');
                }
                else {
                    foreach ($data['products'] as $product) {
                        $em->remove($product);
                    }

                    $em->flush();
                    $this->addTranslatedMessage('flash.bulk_deleted');

                    // Rebuild the query, since some objects are now missing.
                    return $this->redirectToRoute('product_index');
                }
            }
        }

        return $this->render('Product\index.html.twig',
        [
            'products' => $pagination,
            //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
            'currency' => $empresa == null ? 'USD':$empresa->getCurrency(),
            'search_form' => $form->createView(),
            'list_form' => $listForm->createView(),
            'decpunit' => $empresa->getDecPunit(),
        ]);
    }

    /**
     * @Route("/autocomplete-reference", name="product_autocomplete_reference")
     */
    public function autocompleteReferenceAction(EmFactory $emFactory, Request $request): \Symfony\Component\HttpFoundation\Response
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $entities = $em->getRepository(\App\Entity\Model\Product::class)
            ->findLikeReference($request->get('term'), $empresa->getId());

        return new JsonResponse($entities);
    }

    /**
     * @Route("/add", name="product_add")
     *
     */
    public function addAction(EmFactory $emFactory, Request $request, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $product = new Product();

        $form = $this->createForm(\App\Form\ProductType::class, $product, [
            'action' => $this->generateUrl('product_add'),
        ]);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            $aux = $em->getRepository(\App\Entity\Model\Product::class)->findLikeReference($product->getReference(), $empresa->getId());
            $aux = reset($aux);
            if($aux === false) {

                $product->setEmpresa($empresa);
                $em->persist($product);
                $em->flush();
                $this->addTranslatedMessage('flash.added');

                return $this->redirectToRoute('product_index');
            }
            else
                $this->addTranslatedMessage('Referencia del producto ya se encuentra registrado', 'danger');


        }

        return $this->render('Product\edit.html.twig',
        ['form' => $form->createView(), 'entity' => $product]);
    }

    /**
     * @Route("/edit/{slug}prd{id}", name="product_edit")
     *
     */
    public function editAction($id, EmFactory $emFactory, Request $request, $slug, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $product = $em->getRepository(\App\Entity\Model\Product::class)->findBySlug($slug,$id);
        if (!$product) {
            throw $this->createNotFoundException('Unable to find Product entity.');
        }

        $form = $this->createForm(\App\Form\ProductType::class, $product, [
            'action' => $this->generateUrl('product_edit', ['id' => $id, 'slug' => $slug]),
        ]);
        $form->handleRequest($request);

        if ($form->isSubmitted()) {
            if ($request->request->has('Form-delete')) {
                $em->remove($product);
                $em->flush();
                $this->addTranslatedMessage('flash.deleted');

                return $this->redirectToRoute('product_index');
            }

            if ($form->isValid()) {
                $product->setEmpresa($empresa);
                $em->persist($product);
                $em->flush();
                $this->addTranslatedMessage('flash.updated');

                return $this->redirectToRoute('product_edit', ['id' => $id, 'slug' => $slug]);
            }
        }

        return $this->render('Product\edit.html.twig',
            ['form' => $form->createView(), 'entity' => $product]);
    }

    protected function addTranslatedMessage($message, $status = 'success')
    {
        $this->addFlash($status, $this->translator->trans($message, [], 'product'));
    }


    /**
     * @Route("/vnt-search-reference", name="product_vntsearch_reference")
     */
    public function vntSearchReferenceAction(EmFactory $emFactory, Request $request): \Symfony\Component\HttpFoundation\Response
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $limit = 5;

        $aux = $request->request->all();

        $page = 0;

        $start = $page > 1 ? ($page - 1) * $limit : 0;

        $condition = Funciones::cleanParametro('');

        $tipo = Funciones::cleanParametro('product');

        //$entities = $em->getRepository('App\Entity\Model\Inventarios\Product')
        //    ->findxReference($request->get('term'), $empresa->getId());

        if (isset($_GET['sEcho'])) {
            $sEcho = $_GET['sEcho'];
        }
        if (isset($_GET['iDisplayStart'])) {
            $start = intval($_GET['iDisplayStart']);
        }
        if (isset($_GET['iDisplayLength'])) {
            $iDisplayLength = intval($_GET['iDisplayLength']);
        }
        if (isset($_GET['sSearch'])) {
            $condition = Funciones::cleanParametro($_GET['sSearch']);
        }

        $repo = $em->getRepository(Product::class);

        $sql = "'p.id','p.reference','p.description','p.price','p.stock'";
        $paginator = $repo->paginatedAjaxVentas($sql, $condition, $limit, $start, $empresa->getId());

        $total_data = is_countable($paginator) ? count($paginator) : 0;
        $pagination = [];

        foreach ($paginator as $post) {
            //$post = new Product();
            $item = [
                $post->getSlug(),
                $post->getReference(),
                $post->getDescription(),
                $post->getPrice(),
                //'stock' => $post->getClase() === 'SER' ? '*' : $post->getStock(),
            ];

            $aux = $post->getIva();

            if ($aux) {
                //$por = $aux->get;
                $taxaux = $em->getRepository(Tax::class)->findIdTaxesEmpresaPorcentaje($empresa->getId(), $aux);
                if ($taxaux)
                    $item[] = $taxaux->getId();
                else {
                    $item[] = $em->getRepository(Tax::class)->findTaxDefault($empresa->getId());
                    //$tax = $item[];
                }
            } else {
                $tax = $em->getRepository(Tax::class)->findTaxDefault($empresa->getId());
                if($tax)
                    $item[] = $tax->getId();
            }

            $pagination[] = $item;

        }

        $arr = array(
            "iTotalRecords" => $total_data,
            "iTotalDisplayRecords" => $total_data,
            'aaData' => $pagination
        );

        $post_data = json_encode($arr);

        return new Response($post_data, 200, array('Content-Type' => 'application/json'));

    }

    /**
     * @Route("desplantilla", name="product_plantilla_download")
     *
     */
    public function plantilladownloadAction(EmFactory $emFactory, Request $request, TranslatorInterface $translator, PaginatorInterface $paginator)
    {

        $format = 'xls';
        $filename = 'productos' . '.' . $format;

        $contentType = 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet';

        $content = file_get_contents($filename);
        $response = new Response();
        $response->headers->set('Content-Type', $contentType);
        $response->headers->set('Content-Disposition', 'attachment;filename="' . $filename . '"');

        $response->headers->addCacheControlDirective('no-cache', true);
        $response->headers->addCacheControlDirective('must-revalidate', true);

        $response->setContent($content);
        return $response;
    }

    /**
     * @Route("upload", name="product_upload")
     *
     */
    public function uploadAction(EmFactory $emFactory, Request $request, TranslatorInterface $translator, PaginatorInterface $paginator)
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository('App\Entity\Model\Empresa');
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $data["fileproducto"] = '';

        $form = $this->createFormBuilder($data)
            ->add('fileproducto', FileType::class, [
                'required' => true,
                //'attr' => ['style' =>"color: transparent"],
                'label' => 'Archivo Productos',
            ])
            ->setAction($this->generateUrl('product_upload'))
            ->getForm();

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $formData = $form->getData();

            if (!empty($formData['fileproducto'])) {
                /** @var Symfony\Component\HttpFoundation\File\UploadedFile $file */
                $file = $formData['fileproducto'];

                $ext = $file->getClientOriginalName();
                $ext = explode('.', $ext);
                if (strtolower($ext[1]) === 'xls' || strtolower($ext[1]) === 'xls') {

                    // Move the file to the uploads directory.
                    $uploadsDir = "temp/";

                    if (!file_exists($uploadsDir)) {
                        mkdir($uploadsDir, 0755, true);
                    }

                    $fileName = $empresa->getRuc() . '.produ.xls';

                    try {
                        $newFile = $file->move($uploadsDir, $fileName);

                        $spreadsheet = IOFactory::load($uploadsDir . $fileName); // Here we are able to read from the excel file
                        $row = $spreadsheet->getActiveSheet()->removeRow(1); // I added this to be able to remove the first file line
                        $sheetData = $spreadsheet->getActiveSheet()->toArray(null, true, true, true); // here, the read data is turned into an array
                        $añadidos = 0;
                        $repetidos =0;
                        foreach ($sheetData as $Row) {

                            $codigo = $Row['A']; // store the first_name on each iteration
                            $nombre = $Row['B']; // store the last_name on each iteration
                            $precio = $Row['C'];     // store the email on each iteration

                            $repo = $em->getRepository('App\Entity\Model\Product');

                            $producto = $repo->findOneBy(array('reference' => $codigo));
                            if (!$producto) {
                                $producto = new Product();
                                $producto->setReference($codigo);
                                $producto->setDescription($nombre);
                                $producto->setPrice($precio);
                                $producto->setEmpresa($empresa);
                                $em->persist($producto);

                                $añadidos++;
                                // here Doctrine checks all the fields of all fetched data and make a transaction to the database.
                            }
                            else
                                $repetidos++;
                        }
                        $em->flush();
                        $em->clear();

                    } catch (FileException $e) {
                        $this->addTranslatedMessage('ERROR CARGAR ARCHIVO', 'danger');
                    }

                    $this->addTranslatedMessage('Productos insertados: '. $añadidos. PHP_EOL.'Productos codigo repetido: '.$repetidos);
                    return $this->redirect($this->generateUrl('product_index'));
                } else {
                    $this->addTranslatedMessage('Extension archivo no valido', 'warning');
                }

            }
        }

        return $this->render('Product\upload.html.twig',
            array(
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'form' => $form->createView(),
            ));
    }

}
