<?php

namespace App\Controller;

use App\Entity\Model\Empresa;
use App\Entity\Model\Tax;
use App\Form\SearchGenericType;
use App\Service\EmFactory;
use DateTime;
use Doctrine\ORM\EntityManagerInterface;
use Knp\Component\Pager\PaginatorInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use App\Entity\Model\Invoice;

class DashboardController extends AbstractController
{
    protected $translator;

    /**
     * @Route("/dashboard", name="dashboard_index")
     *
     */
    public function indexAction(EmFactory $emFactory, Request $request, PaginatorInterface $paginator): \Symfony\Component\HttpFoundation\Response
    {
        $alerta = [];
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $invoiceRepo = $em->getRepository(Invoice::class);
        $invoiceRepo->setPaginator($paginator);

        $taxRepo = $em->getRepository(Tax::class);
        $taxes = $taxRepo->findTaxesEmpresa($empresa->getId());

        $form = $this->createForm(SearchGenericType::class, null, [
            'action' => $this->generateUrl('dashboard_index'),
            'method' => 'GET',
        ]);
        $form->handleRequest($request);

        $params = $form->isSubmitted() && $form->isValid() ? $form->getData() : [];

        // Last invoices.
        // @todo Unhardcode this.
        $limit = 5;
        $pagination = $invoiceRepo->paginatedSearch($params, $limit, $request->query->getInt('page', 1), $empresa->getId());
        $totals = $invoiceRepo->getTotals($params, $empresa->getId());

        // Last overdue invoices.
        $overdueParams = $params;
        $overdueParams['status'] = Invoice::OVERDUE;
        // @todo Unhardcode this.
        $limit = 50;
        $paginationDue = $invoiceRepo->paginatedSearch($overdueParams, $limit, $request->query->getInt('page', 1), $empresa->getId());
        $totalsDue = $invoiceRepo->getTotals($overdueParams, $empresa->getId());
        $totals['overdue'] = $totalsDue['due'];

        // Tax totals.
        foreach ($taxes as $tax) {
            $taxId = $tax->getId();
            $params['tax'] = $taxId;
            $taxTotals = $invoiceRepo->getTotals($params, $empresa->getId());
            $totals['taxes'][$taxId] = $taxTotals['tax_' . $taxId];
        }

       $contratados = $empresa->getCmpContratados();

        $emitidos = $this->electroEmitidos($empresa->getId(), $em);

        $alertas = null;
        $firma = $this->caducaFirma($emFactory);

        if ($firma != null) {
            $alerta['prioridad'] = $firma['prioridad'];
            $alerta['mensaje'] = '<p>Firma caduca el: ' . $firma['caduca'] . '</p><p>Recuerde renovar o adquirir una nueva firma con la autoridad certificadora</p>';

            $alertas[] = $alerta;
        }

        $caduca = $this->caducaPlan($emFactory);

        if ($caduca != null) {
            $alerta['prioridad'] = $caduca['prioridad'];
            $alerta['mensaje'] = $caduca['caduca'];

            $alertas[] = $alerta;
        }

        return $this->render('Dashboard\index.html.twig',
            [
                'invoices' => $pagination,
                'overdue_invoices' => $paginationDue,
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $empresa == null ? 'USD' : $empresa->getCurrency(),
                'search_form' => $form->createView(),
                'totals' => $totals,
                'taxes' => $taxes,
                'paginatable' => false,
                'sortable' => false,
                'contratados' => $contratados == null ? 0 : $contratados,
                'emitidos' => $emitidos == null ? 0 : $emitidos,
                'alerts' => $alertas
            ]);
    }

    private function electroContratados($Empresa_id, EntityManagerInterface $entityManager)
    {

        $em = $entityManager;

        return $em->getRepository(\App\Entity\Model\Comprobantes::class)->countRecibidos($Empresa_id);
    }

    private function electroEmitidos($Empresa_id, EntityManagerInterface $entityManager)
    {

        $em = $entityManager;
        $facturas = $em->getRepository(\App\Entity\Model\Invoice::class)->countEmitidos($Empresa_id);

        $retenciones = $em->getRepository(\App\Entity\Model\Retencion::class)->countEmitidos($Empresa_id);

        return $facturas + $retenciones;
    }

    private function caducaFirma(EmFactory $emFactory)
    {
        $firma = null;
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        try {
            $fec_firma = $empresa->getFechaCaducaFirma();

            if($fec_firma == null)
                return null;

            $date1 = $fec_firma;
            $date2 = new DateTime("now");

            $diff = $date1->diff($date2);
            //$dias= $this->get_format($diff);

            $dias = $diff->days;

            if ($dias <= 30) {
                if ($dias >= 20 && $dias <= 30) {
                    $firma['prioridad'] = 'warning';
                } elseif ($dias < 20) {
                    $firma['prioridad'] = 'danger';
                }

                $firma['caduca'] = $fec_firma->format('d M Y');
            }

            return $firma;
        }
        catch (\Exception $exception){
            return  null;
        }

    }

    private function caducaPlan(EmFactory $emFactory)
    {
        $firma = null;
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        try {
            $fec_caduca = $empresa->getFechaCaduca();

            if($fec_caduca == null)
                return null;

            $date1 = $fec_caduca;
            $date2 = new DateTime("now");

            if($date1 < $date2){
                $firma['prioridad'] = 'danger';
                $firma['caduca'] = 'Tu plan ha caducado.'.'<p>Renueve tu plan o adquiera uno nuevo.</p>';
                return $firma;
            }

            $diff = $date1->diff($date2);
            //$dias= $this->get_format($diff);

            $dias = $diff->days;

            if($dias === 0){
                $horas = $diff->h;
                $minutos = $diff->i;
                $firma['prioridad'] = 'danger';
                $firma['caduca'] = 'Tu plan caduca en '.$horas.' horas '.$minutos.' minutos ';
            }
            else if ($dias <= 10) {
                if ($dias >= 5 && $dias <= 10) {
                    $firma['prioridad'] = 'warning';
                } elseif ($dias < 5) {
                    $firma['prioridad'] = 'danger';
                }

                //$firma['caduca'] = $fec_firma->format('d M Y');
                $firma['caduca'] = 'Tu plan caduca en '.$dias.' dias';

            }

            return $firma;
        }
        catch (\Exception $exception){
            return  null;
        }

    }

}
