<?php

namespace App\Entity\Model;

use App\Util\Inflector;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\Common\Collections\ArrayCollection;
use Gedmo\Timestampable\Traits\TimestampableEntity;
use Gedmo\Timestampable\Timestampable;
use Symfony\Component\Validator\Constraints as Assert;
use function Symfony\Component\String\u;

/**
 * App\Entity\Model\AbstractGuiaRemision
 *
 * @ORM\MappedSuperclass
 * @ORM\HasLifecycleCallbacks
 */
abstract class AbstractGuiaRemision implements Timestampable
{
    /**
     * @var \Doctrine\Common\Collections\ArrayCollection|\App\Entity\Model\Destinatario[]|mixed|mixed[]
     */
    public $destinatarios;

    /**
     * @var string|mixed
     */
    public $contact_person;

    public $empresa;

    use TimestampableEntity;

    /**
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private  $id = null;

    /**
     * @ORM\Column(name="slug", type="string", length=32, nullable=true, unique=true)
     */
    private  $slug = null;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Model\Transportista", inversedBy="guiaremisions")
     * @ORM\JoinColumn(name="transportista_id", referencedColumnName="id")
     */
    protected $transportista;

    /**
     *
     * @ORM\Column(name="transportista_name", type="string", length=255)
     * @Assert\NotBlank(message = "Ingrese el nombre del transportista")
     */
    private  $transportista_name = null;

    /**
     *
     * @ORM\Column(name="transportista_identification", type="string", length=128, nullable=true)
     * @Assert\NotBlank(message = "Ingrese el número de identificación del transportista")
     */
    private  $transportista_identification = null;

    /**
     *
     * @ORM\Column(name="transportista_email", type="string", length=255)
     * @Assert\NotBlank(message = "Ingrese el correo electrónico del Transportista")
     * @Assert\Email()
     */
    private  $transportista_email = null;

    /**
     * @var text $direccion_partida
     *
     * @ORM\Column(name="direccion_partida", type="text", nullable=true)
     */
    private $direccion_partida = null;

    /**
     * @ORM\Column(name="transportista_placa", type="string", length=10, nullable=true)
     */
    private  $transportista_placa = null;

    /**
     * @var text $notes
     *
     * @ORM\Column(name="notes", type="text", nullable=true, length=200)
     */
    private $notes = null;

    /**
     * @var text $obs
     *
     * @ORM\Column(name="note1", type="text", nullable=true, length=200)
     */
    private $note1 = null;

    /**
     * @var smallint $status
     *
     * @ORM\Column(name="status", type="smallint", nullable=true)
     * @Assert\Length(min=0, max=3)
     */
    protected $status = 0;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Model\Series")
     * @ORM\JoinColumn(name="series_id", referencedColumnName="id")
     * @Assert\NotBlank()
     *
     * unidirectional many-to-one
     */
    private $series;

    /**
     * @ORM\Column(type="integer", nullable=true)
     */
    private  $usuario = null;

    public function __construct()
    {
        $this->destinatarios = new ArrayCollection();
    }

    /**
     * Get id
     *
     * @return integer
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set transportista parameters from transportista entity.
     *
     * @param App\Entity\Model\Transportista $transportista
     */
    public function setFromTransportista(Transportista $transportista)
    {
        if (empty($this->transportista_name) && $transportista->getName()) {
            $this->transportista_name = $transportista->getName();
        }

        if (empty($this->transportista_email) && $transportista->getEmail()) {
            $this->transportista_email = $transportista->getEmail();
        }

        if (empty($this->transportista_identification) && $transportista->getIdentification()) {
            $this->transportista_identification = $transportista->getIdentification();
        }

        if (empty($this->transportista_placa) && $transportista->getPlaca()) {
            $this->transportista_placa = $transportista->getPlaca();
        }
    }

    /**
     * Set transportista.
     *
     * @param App\Entity\Model\Transportista $transportista
     */
    public function setTransportista(Transportista $transportista)
    {
        $this->transportista = $transportista;
    }

    /**
     * @return mixed
     */
    public function getTransportista()
    {
        return $this->transportista;
    }

    /**
     * Set transportista_name
     *
     * @param string $transportistaName
     */
    public function setTransportistaName($transportistaName)
    {
        $this->transportista_name = $transportistaName;
    }

    /**
     * Get transportista_name
     *
     * @return string
     */
    public function getTransportistaName()
    {
        return $this->transportista_name;
    }

    /**
     * Set transportista_identification
     *
     * @param string $transportistaIdentification
     */
    public function setTransportistaIdentification($transportistaIdentification)
    {
        $this->transportista_identification = $transportistaIdentification;
    }

    /**
     * Get transportista_identification
     *
     * @return string
     */
    public function getTransportistaIdentification()
    {
        return $this->transportista_identification;
    }

    /**
     * Set transportista_email
     *
     * @param string $transportistaEmail
     */
    public function setTransportistaEmail($transportistaEmail)
    {
        $this->transportista_email = $transportistaEmail;
    }

    /**
     * Get transportista_email
     *
     * @return string
     */
    public function getTransportistaEmail()
    {
        return $this->transportista_email;
    }

    /**
     * Set direccion_partida
     *
     * @param text $direccionpPartida
     */
    public function setDireccionpPartida($direccionpPartida)
    {
        $this->direccion_partida = $direccionpPartida;
    }

    /**
     * Get direccion_partida
     *
     * @return text
     */
    public function getDireccionpPartida()
    {
        return $this->direccion_partida;
    }

    /**
     * @return string
     */
    public function getTransportistaPlaca()
    {
        return $this->transportista_placa;
    }

    /**
     * @param string $transportista_placa
     */
    public function setTransportistaPlaca($transportista_placa)
    {
        $this->transportista_placa = $transportista_placa;
    }


    /**
     * Set contact_person
     *
     * @param string $contactPerson
     */
    public function setContactPerson($contactPerson)
    {
        $this->contact_person = $contactPerson;
    }

    /**
     * Get contact_person
     *
     * @return string
     */
    public function getContactPerson()
    {
        return $this->contact_person;
    }


    /**
     * Set notes
     *
     * @param text $notes
     */
    public function setNotes($notes)
    {
        $this->notes = strtoupper($notes);
    }

    /**
     * Get notes
     *
     * @return text
     */
    public function getNotes()
    {
        return $this->notes;
    }


    /**
     * Set status
     *
     * @param integer $status
     */
    public function setStatus($status)
    {
        $this->status = $status;
    }

    /**
     * Get status
     *
     * @return integer
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * Set serie
     *
     * @param App\Entity\Model\Series $series
     */
    public function setSeries(Series $series)
    {
        $this->series = $series;
    }

    /**
     * Get serie
     *
     * @return App\Entity\Model\Series
     */
    public function getSeries()
    {
        return $this->series;
    }


    /** ########### CUSTOM METHODS ################## */


    public function __isset($name)
    {
        return array_key_exists($name, get_object_vars($this));
    }

    /**
     * Add destinatarios
     *
     * @param App\Entity\Model\Destinatario $destinatario
     */
    public function addDestinatario(Destinatario $destinatario)
    {
        $this->destinatarios[] = $destinatario;
    }

    /**
     * Get items
     *
     * @return Doctrine\Common\Collections\Collection
     */
    public function getDestinatarios()
    {
        return $this->destinatarios;
    }

    /**
     * removeItem
     * removes an item and recalculcates amounts
     *
     * @param mixed $mixed : can be an integer or an item instance
     *                       - if an integer, removes the item with
     *                         that position in the collection
     *                       - if an instance, removes that item
     * @author JoeZ99 <jzarate@gmail.com>
     */
    public function removeDestinatarios($mixed)
    {
        if ($mixed instanceof Destinatario) {
            $items = $this->getDestinatarios();
            foreach ($items as $ref => $item) {
                if ($item === $mixed) {
                    unset($items[$ref]);
                    break;
                }
            }
        } elseif (is_int($mixed)) {
            unset($this->destinatarios[$mixed]);
        }

    }

    /* ** OTHER ** */

    private $decimals = null;

    private function getDecimals()
    {
        if (!$this->decimals) {
            $this->decimals = 2;
        }

        return $this->decimals;
    }


    /**
     * calculate values over items
     *
     * Warning!! this method only works when called from a real entity, not
     * the abstract.
     *
     * @param string $field
     * @param boolean $rounded
     * @return float
     */
    public function calculate($field, $rounded = false)
    {
        $val = 0;
        if ($field == 'paid_amount') {
        }

        if ($rounded) {
            return round($val, $this->getDecimals());
        }

        return $val;
    }

    /** *********** LIFECYCLE CALLBACKS ************* */

    /**
     * @ORM\PreUpdate
     * @ORM\PrePersist
     */
    public function preSave(LifecycleEventArgs $args)
    {
        $this->checkStatus();
        $transportistas = $args->getObjectManager()->getRepository(Transportista::class)->findBy([
            'name' => $this->getTransportistaName(),
            'identification' => $this->getTransportistaIdentification(),
            'empresa' => $this->getEmpresa(),
        ]);
        $transportista = reset($transportistas);
        if ($transportista) {
            $this->setTransportista($transportista);
        }

        foreach ($this->getDestinatarios() as $destinatario) {
            //$destinatario = new Destinatario();
            //$customer = new Customer();
            $customers = $args->getObjectManager()->getRepository(Customer::class)->findBy([
                'name' => $destinatario->getCustomerName(),
                'identification' => $destinatario->getCustomerIdentification(),
                'empresa' => $this->getEmpresa(),
            ]);
            $customer = reset($customers);
            if ($customer) {
                $destinatario->setCustomer($customer);
            }

            foreach ($destinatario->getItems() as $itemguia)
                $itemguia->setGuiaRemision($this);
        }

        if($this->id === null)
            $this->slug = md5('sv_'.$this->empresa->getId().uniqid());

    }

    /**
     * @return mixed
     */
    public function getUsuario()
    {
        return $this->usuario;
    }

    /**
     * @param mixed $usuario
     */
    public function setUsuario($usuario)
    {
        $this->usuario = $usuario;
    }

    /**
     * @return string
     */
    public function getSlug(): string
    {
        return $this->slug;
    }

    /**
     * @return text
     */
    public function getNote1()
    {
        return $this->note1;
    }

    /**
     * @param text $note1
     */
    public function setNote1($note1)
    {
        $this->note1 = $note1;
    }

}
